package ru.yandex.qloud.kikimr.search;

import org.apache.commons.lang3.StringUtils;
import ru.yandex.qloud.kikimr.lucene.ESQueryConverter;
import ru.yandex.qloud.kikimr.lucene.PagingParameters;
import ru.yandex.qloud.kikimr.lucene.TimeRangeFilter;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class KikimrYqlQueryBuilder {

    private static final ESQueryConverter ES_QUERY_CONVERTER = new ESQueryConverter();
    private static final YqlQueryConverter YQL_QUERY_CONVERTER = new YqlQueryConverter();

    public static String buildYqlQuery(
            @Nonnull String table,
            @Nonnull QueryWhereCondition queryWhereCondition,
            @Nullable PagingParameters pagingParameters,
            @Nonnull TimeRangeFilter timeRangeFilter,
            boolean accessLog
    ) {
        return String.format(
                "PRAGMA kikimr.IsolationLevel='ReadUncommitted'; SELECT * FROM [%s] WHERE %s %s LIMIT %d;",
                table,
                convertToYqlWhereCondition(queryWhereCondition, pagingParameters, timeRangeFilter, accessLog),
                getOrderBy(pagingParameters),
                pagingParameters != null ? pagingParameters.getLimit() : PagingParameters.DEFAULT_LIMIT
        );
    }

    public static String convertToYqlWhereCondition(QueryWhereCondition queryWhereCondition, PagingParameters pagingParameters, TimeRangeFilter timeRangeFilter, boolean accessLog) {
        StringBuilder converted = new StringBuilder();
        if (StringUtils.isNotBlank(queryWhereCondition.getYqlQueryFilters())) {
            converted.append(YQL_QUERY_CONVERTER.convertToYqlWhereCondition(queryWhereCondition.getYqlQueryFilters(), accessLog));
        }

        if (StringUtils.isNotBlank(queryWhereCondition.getEsQueryFilters()) || converted.length() == 0) {
            if (converted.length() > 0) {
                converted.append(" AND ");
            }
            converted.append(ES_QUERY_CONVERTER.convertToYqlWhereCondition(queryWhereCondition.getEsQueryFilters(), accessLog));
        }

        String timeRangeCondition = convertTimeRangeCondition(timeRangeFilter);
        if (StringUtils.isNoneBlank(timeRangeCondition)) {
            converted.append(" AND ").append(timeRangeCondition);
        }

        if (pagingParameters != null) {
            converted.append(" AND ").append(getPagingWhereCondition(pagingParameters));
        }

        return converted.toString();
    }

    private static String getPagingWhereCondition(@Nonnull PagingParameters pagingParameters) {
        String comparisonSign = pagingParameters.getPagingDirection() == PagingParameters.PagingDirection.PREV ? ">" : "<";
        return String.format(
                "(timestamp %1$2s %2$2d OR timestamp = %2$2d AND qloud_instance %1$2s %3$2s "
                        + "OR timestamp = %2$2d AND qloud_instance = %3$2s AND pushclient_row_id %1$2s %4$2d)",
                comparisonSign, pagingParameters.getTimestamp(),
                "'" + pagingParameters.getQloudInstance() + "'",
                pagingParameters.getPushClientId()
        );
    }

    @Nonnull
    private static String getOrderBy(PagingParameters pagingParameters) {
        String sortDirection = pagingParameters == null || pagingParameters.getPagingDirection() == PagingParameters.PagingDirection.PREV ? "ASC" : "DESC";
        return String.format("ORDER BY timestamp %1$2s, qloud_instance %1$2s, pushclient_row_id %1$2s", sortDirection);
    }

    private static String convertTimeRangeCondition(TimeRangeFilter timeRangeFilter) {
        StringBuilder converted = new StringBuilder();
        if (timeRangeFilter.getTimestampFrom() != null) {
            converted.append("(timestamp <= -").append(timeRangeFilter.getTimestampFrom());
        }
        if (timeRangeFilter.getTimestampTo() != null) {
            if (converted.length() > 0) {
                converted.append(" AND ");
            } else {
                converted.append("(");
            }
            converted.append("timestamp >= -").append(timeRangeFilter.getTimestampTo());
        }
        if (converted.length() > 0) {
            converted.append(")");
        }
        return converted.toString();
    }

    private static class YqlQueryConverter implements QueryWhereConverter {

        @Override
        public String convertToYqlWhereCondition(String query, boolean isAccessLog) {
            return StringUtils.isBlank(query) ? "TRUE" : ("(" + query + ")");
        }

    }
}
