package ru.yandex.qloud.kikimr.services;

import com.codahale.metrics.MetricRegistry;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.rs.security.cors.CrossOriginResourceSharing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import ru.yandex.qloud.kikimr.lucene.ESResultConverter;
import ru.yandex.qloud.kikimr.lucene.PagingParameters;
import ru.yandex.qloud.kikimr.lucene.TimeRangeFilter;
import ru.yandex.qloud.kikimr.scheme.TableStatistics;
import ru.yandex.qloud.kikimr.search.KikimrQueryRequest;
import ru.yandex.qloud.kikimr.search.KikimrQueryRequestFactory;
import ru.yandex.qloud.kikimr.search.QueryWhereCondition;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;
import ru.yandex.qloud.kikimr.transport.YQL;
import ru.yandex.qloud.kikimr.utils.TableUtils;

import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.MoreObjects.firstNonNull;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_ACCESS_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_APPLICATION_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_DATE_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_ENVIRONMENT_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_PROJECT_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterUtils.validateTableParameters;

/**
 * @author violin
 */
@Service("logProxyService")
@Consumes(MediaType.WILDCARD)
@Produces(MediaType.APPLICATION_JSON)
@CrossOriginResourceSharing(allowAllOrigins = true, allowCredentials = true)
@Path("/")
public class LogProxyService {
    private final static Logger LOG = LoggerFactory.getLogger(LogProxyService.class);

    private static final int TIMEOUT_THRESHOLD = 60000;

    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ISO_DATE_TIME;

    @Inject
    private KikimrQueryRequestFactory kikimrQueryRequestFactory;
    @Inject
    private YQL yql;
    @Inject
    private ESResultConverter esResultConverter;
    @Inject
    private KikimrScheme kikimrScheme;

    @Inject
    private MetricRegistry metricRegistry;

    @GET
    @Path("/search")
    public ESResultConverter.ESResult getLog(
            @QueryParam(Q_PROJECT_PARAM) String qloudProject,
            @QueryParam(Q_APPLICATION_PARAM) String qloudApplication,
            @QueryParam(Q_ENVIRONMENT_PARAM) String qloudEnvironment,
            @QueryParam(Q_DATE_PARAM) String qloudDate,
            @QueryParam(Q_ACCESS_PARAM) @DefaultValue("false") boolean accessLog,
            @QueryParam("q") String esQuery,
            @QueryParam("from") String pagingKey,
            @QueryParam("rev") String pagingDirection, //reverse: true or false
            @QueryParam("time-range") String timeRange,
            @QueryParam("yql") String yqlQuery

    ) {
        validateTableParameters(qloudProject, qloudApplication, qloudEnvironment, qloudDate);
        metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "requests")).inc();

        String path = TableUtils.toKikimrNode(qloudProject, qloudApplication, qloudEnvironment);
        PagingParameters pagingParameters = parsePagingParameters(pagingKey, pagingDirection);
        TimeRangeFilter timeRangeFilter = parseTimeRangeFilter(timeRange, qloudDate);

        final long startTime = System.currentTimeMillis();

        long time;
        KikimrQueryRequest kikimrQueryRequest = null;
        try {
             kikimrQueryRequest = kikimrQueryRequestFactory.createKikimrQueryRequest(
                 getQueryCondition(esQuery, yqlQuery), path, pagingParameters, timeRangeFilter, qloudDate, accessLog
            );
            YQL.Result result = yql.queries(kikimrQueryRequest.getKikimrQueries(), kikimrQueryRequest.getMaxResultsCount());
            time = System.currentTimeMillis() - startTime;
            String status = time < TIMEOUT_THRESHOLD ? "OK_YQL" : "TIMEOUT_YQL";
            LOG.debug(
                "{} {} ms; kikimr_query = {}; result count = {}",
                status, time, kikimrQueryRequest, result.getRows().size()
            );

            metricRegistry.timer(MetricRegistry.name(LogProxyService.class, "search", "success")).update(time, TimeUnit.MILLISECONDS);

            return esResultConverter.convert(result);
        } catch (Exception e) {
            time = System.currentTimeMillis() - startTime;
            LOG.debug(
                    String.format(
                            "FAIL_YQL %d ms; path = %s; kikimr_query = %s; result count = 0; error message = %s",
                            time, path,
                            firstNonNull(kikimrQueryRequest, kikimrQueryRequestFactory.createEmptyQueryRequest(getQueryCondition(esQuery, yqlQuery), pagingParameters, timeRangeFilter)).toString(),
                            ExceptionUtils.getRootCauseMessage(e)
                    ),
                    e
            );

            metricRegistry.timer(MetricRegistry.name(LogProxyService.class, "search", "fail")).update(time, TimeUnit.MILLISECONDS);

            throw toLogSearchException(e, kikimrQueryRequest);
        }
    }

    private QueryWhereCondition getQueryCondition(String esQueryFilters, String yqlQueryFilters) {
        return new QueryWhereCondition(esQueryFilters, yqlQueryFilters);
    }

    @Nullable
    private PagingParameters parsePagingParameters(String pagingKey, String pagingDirection) {
        if (StringUtils.isBlank(pagingKey) || StringUtils.isBlank(pagingDirection)) {
            return null;
        }
        String[] keyParts = pagingKey.split("\\|");
        if (keyParts.length != 3) {
            throw new IllegalArgumentException("'from' parameter is incorrect: " + pagingKey);
        }
        return new PagingParameters(
                Long.parseLong(keyParts[0]),
                keyParts[1],
                Long.parseLong(keyParts[2]),
                "true".equals(pagingDirection) ? PagingParameters.PagingDirection.NEXT : PagingParameters.PagingDirection.PREV
        );
    }

    private TimeRangeFilter parseTimeRangeFilter(String rangeParameter, String qloudDate) {
        Long timestampFrom = null;
        Long timestampTo = null;

        if (StringUtils.isNotBlank(rangeParameter)) {
            if (! rangeParameter.contains("-")) {
                throw new IllegalArgumentException("time-range parameter invalid: " + rangeParameter);
            }
            int splitIndex;
            int hyphenCount = StringUtils.countMatches(rangeParameter, "-");
            if (hyphenCount == 1) {
                splitIndex = rangeParameter.indexOf("-");
            } else if (rangeParameter.startsWith("-")) {
                splitIndex = 0;
            } else {
                splitIndex = StringUtils.ordinalIndexOf(rangeParameter, "-", hyphenCount <= 5 ? 3 : 4);
            }

            timestampFrom = parseTimestamp(qloudDate, StringUtils.substring(rangeParameter, 0, splitIndex));
            timestampTo = parseTimestamp(qloudDate, StringUtils.substring(rangeParameter, splitIndex + 1));

            if (timestampFrom != null && timestampTo != null && timestampFrom > timestampTo) {
                throw new IllegalArgumentException(String.format("time range incorrect: 'from' is after 'to' (%s)", rangeParameter));
            }
        }

        return new TimeRangeFilter(timestampFrom, timestampTo);
    }

    private Long parseTimestamp(String date, String time) {
        if (StringUtils.isNotBlank(time)) {
            String timeToParse = time.contains("T") ? time : date + "T" + time;
            if (timeToParse.length() > 20) {
                return DATE_TIME_FORMATTER.parse(timeToParse, OffsetDateTime::from).toInstant().toEpochMilli();
            } else {
                return DATE_TIME_FORMATTER.parse(timeToParse, LocalDateTime::from).toInstant(ZoneOffset.UTC).toEpochMilli();
            }
        }
        return null;
    }

    private LogSearchException toLogSearchException(Exception e, KikimrQueryRequest kikimrQueryRequest) {
        if (e.getMessage() == null) {
            return new LogSearchException("unknown error");
        }

        StringBuilder text = new StringBuilder();
        if (e.getMessage().contains("Table not found") || e.getMessage().contains("Cannot find table")) {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "not_found")).inc();
            text.append("No logs found");
        } else if (e.getMessage().contains("DEADLINE_EXCEEDED")) {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "deadline")).inc();
            text.append("Timeout");
            String statMessage = getTableStatisticsMessage(kikimrQueryRequest);
            if (StringUtils.isNotBlank(statMessage)) {
                text.append(" by searching in ").append(statMessage).append(" logs");
            }
            text.append(". Please limit time range or just retry");
        } else if (e.getMessage().contains("Type annotation")) {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "incorrect_member")).inc();
            text.append("Query is incorrect.");
            if (e.getMessage().contains("Member not found")) {
                for (String line : e.getMessage().split("\n")) {
                    if (line.startsWith("Member not found")) {
                        text.append(" Unknown field : ").append(StringUtils.substringAfter(line, ":")).append(".");
                    }
                }
            }
        } else if (e.getMessage().contains("Unexpected token absence")) {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "incorrect_query")).inc();
            text.append("Query is incorrect.");
            for (String line : e.getMessage().split("\n")) {
                if (line.startsWith("Unexpected token absence")) {
                    text.append(" ").append(StringUtils.substringAfter(line, ":")).append(".");
                }
            }
        } else if (e.getMessage().contains("Unable to parse lucene query")) {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "fail_parse_lucene")).inc();
            text.append("Invalid query, unable to parse");
        } else {
            metricRegistry.counter(MetricRegistry.name(LogProxyService.class, "search", "fail_common")).inc();
            text.append(e.getMessage());
        }
        throw new LogSearchException(text.toString());
    }

    private String getTableStatisticsMessage(KikimrQueryRequest kikimrQueryRequest) {
        if (kikimrQueryRequest == null) {
            return "";
        }

        try {
            long rowCount = 0;
            long bytesCount = 0;

            for (String table : kikimrQueryRequest.getTables()) {
                TableStatistics statistics = kikimrScheme.getTableStatistics(table);
                if (statistics != null) {
                    rowCount += statistics.getRowCount();
                    bytesCount += statistics.getDataSize();
                }
            }
            return String.format("%d lines, %s", rowCount, FileUtils.byteCountToDisplaySize(bytesCount));

        } catch (Exception e) {
            LOG.warn("exception by getting table statistics for " + kikimrQueryRequest.getTables(), e);
        }
        return "";
    }

}
