package ru.yandex.qloud.kikimr.services;

import org.apache.cxf.rs.security.cors.CrossOriginResourceSharing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import ru.yandex.qloud.kikimr.scheme.TableStatistics;
import ru.yandex.qloud.kikimr.transport.KikimrScheme;
import ru.yandex.qloud.kikimr.transport.PathNotFoundException;
import ru.yandex.qloud.kikimr.utils.TableUtils;

import javax.inject.Inject;
import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import java.util.Collections;
import java.util.List;

import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_ACCESS_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_APPLICATION_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_DATE_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_ENVIRONMENT_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterNames.Q_PROJECT_PARAM;
import static ru.yandex.qloud.kikimr.services.ParameterUtils.validateTableParameters;

/**
 * @author violin
 */
@Service("schemeService")
@Consumes(MediaType.WILDCARD)
@Produces(MediaType.APPLICATION_JSON)
@CrossOriginResourceSharing(allowAllOrigins = true, allowCredentials = true)
@Path("/scheme")
public class SchemeService {
    private final static Logger LOG = LoggerFactory.getLogger(SchemeService.class);

    @Inject
    private KikimrScheme kikimrScheme;

    @GET
    @Path("stat")
    public TableStatistics getTableStatistics(
            @QueryParam(Q_PROJECT_PARAM) String qloudProject,
            @QueryParam(Q_APPLICATION_PARAM) String qloudApplication,
            @QueryParam(Q_ENVIRONMENT_PARAM) String qloudEnvironment,
            @QueryParam(Q_DATE_PARAM) String qloudDate,
            @QueryParam(Q_ACCESS_PARAM) @DefaultValue("false") boolean access
    ) {
        validateTableParameters(qloudProject, qloudApplication, qloudEnvironment, qloudDate);
        return kikimrScheme.getTableStatistics(
                TableUtils.toKikimrTable(qloudProject, qloudApplication, qloudEnvironment, qloudDate, access)
        );
    }

    @GET
    @Path("tables")
    public List<String> getTables(
        @QueryParam(Q_PROJECT_PARAM) String qloudProject,
        @QueryParam(Q_APPLICATION_PARAM) String qloudApplication,
        @QueryParam(Q_ENVIRONMENT_PARAM) String qloudEnvironment
    ) {
        validateTableParameters(qloudProject, qloudApplication, qloudEnvironment, null);
        try {
            return kikimrScheme.getAvailableTablesSimpleNames(
                    TableUtils.toKikimrNode(qloudProject, qloudApplication, qloudEnvironment)
            );
        } catch (Exception e) {
            if (e.getCause() instanceof PathNotFoundException) {
                LOG.warn(String.format("path not found for %s", TableUtils.toKikimrNode(qloudProject, qloudApplication, qloudEnvironment)), e);
                return Collections.emptyList();
            } else {
                throw e;
            }
        }
    }
}
