package ru.yandex.qloud.kikimr.transport;

import NKikimrClient.TGRpcServerGrpc;
import com.google.common.base.Strings;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.kikimr.proto.Msgbus;
import ru.yandex.qe.spring.RequestIdentity;
import ru.yandex.qe.spring.servlet.SecurityFilter;

import javax.annotation.Nullable;
import javax.annotation.PreDestroy;
import javax.servlet.http.Cookie;
import java.util.concurrent.TimeUnit;

/**
 * @author violin
 */
@Component
public class KikimrRpc {
    private final static Logger LOG = LoggerFactory.getLogger(KikimrRpc.class);

    private final ManagedChannel channel;
    private final TGRpcServerGrpc.TGRpcServerBlockingStub rpc;
    private final long deadlineMillis;
    private final String rootUserToken;

    public KikimrRpc(
            @Value("${env.KIKIMR_CLUSTER_HOST:qloud-kikimr.yandex.net}") String host,
            @Value("${env.KIKIMR_RPC_PORT:2135}") int port,
            @Value("${env.KIKIMR_DEADLINE_MILLIS:60000}") long deadlineMillis,
            @Value("${env.QLOUD_KIKIMR_ROBOT_YQL_OAUTH_TOKEN:}") String rootUserToken
    ) {
        channel = ManagedChannelBuilder
                .forAddress(host, port)
                .usePlaintext()
                .build();

        rpc = TGRpcServerGrpc.newBlockingStub(this.channel);

        this.deadlineMillis = deadlineMillis;
        this.rootUserToken = rootUserToken;
    }

    public TGRpcServerGrpc.TGRpcServerBlockingStub getRpcWithDeadline() {
        return rpc.withDeadlineAfter(deadlineMillis, TimeUnit.MILLISECONDS);
    }

    @Nullable
    public String whoami() {
        String token = getUserSecurityToken();
        if (token == null) {
            return null;
        }
        Msgbus.TWhoAmI tWhoAmI = Msgbus.TWhoAmI.newBuilder().setSecurityToken(token).build();
        Msgbus.TResponse response = rpc.whoAmI(tWhoAmI);
        return response.hasUserName() ? response.getUserName() : null;
    }

    @Nullable
    protected String getUserSecurityToken() {
        RequestIdentity identity = RequestIdentity.get();

        Cookie sessionCookie = identity.getSessionCookie();
        String userip = identity.getRemoteAddress();
        if (sessionCookie != null && !StringUtils.isBlank(sessionCookie.getValue())) {
            return String.format("blackbox?method=sessionid&userip=%s&host=yandex-team.ru&sessionid=%s", userip, sessionCookie.getValue());
        }

        String oauthToken = Strings.nullToEmpty(
                StringUtils.substringAfter(identity.getHeader(SecurityFilter.AUTHORIZATION_HEADER), "OAuth")
        ).trim();
        if (StringUtils.isNotBlank(oauthToken)) {
            return oauthToken;
        }

        throw new IllegalArgumentException("anonymous requests not allowed here");
    }

    protected String getRootUserToken() {
        return rootUserToken;
    }

    @PreDestroy
    public void shutdown() {
        try {
            channel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            throw new RuntimeException("Can't shutdown gRPC channel", e);
        }
    }
}
