#!/bin/env python3
# -*- coding: utf-8 -*-

import requests
import urllib3
from collections import namedtuple

from pymongo import MongoClient


urllib3.disable_warnings()

MongoReplicaSetTemplate = namedtuple('MongoReplicaSetTemplate', [
    'discovery',
    'replica_set_name'
])

MongoReplicaSet = namedtuple('MongoReplicaSet', [
    'addresses',
    'replica_set_name'
])

NannyMongoService = namedtuple('NannyMongoService', [
    'dcs',
    'installation',
])

MongoDiscovery = namedtuple('MongoDiscovery', [
    'static',
    'nanny_service'
])

NANNY_URL = 'https://nanny.yandex-team.ru'

LOCATIONS = {
    'test': MongoReplicaSetTemplate(
        discovery=MongoDiscovery(
            static=[
                'sas1-2642-sas-qloud-mongo-test-19779.gencfg-c.yandex.net:19779',
                'vla1-0470-vla-qloud-mongo-test-21051.gencfg-c.yandex.net:21051',
                'man2-2284-439-man-qloud-mongo-test-21902.gencfg-c.yandex.net:21902'
            ],
            nanny_service=NannyMongoService(
                dcs=['man', 'sas', 'vla'],
                installation='testing'
            )),
        replica_set_name='test-root'
    ),
    'pre': MongoReplicaSetTemplate(
        discovery=MongoDiscovery(
            static=[
                'sas1-1578-bce-sas-qloud-mongo--460-19216.gencfg-c.yandex.net:19216',
                'iva1-1029-a1d-msk-iva-qloud-mo-2bd-10782.gencfg-c.yandex.net:10782',
                'myt1-0317-104-msk-myt-qloud-mo-e44-10792.gencfg-c.yandex.net:10792'],
            nanny_service=NannyMongoService(
                dcs=['iva', 'myt', 'sas'],
                installation='prestable'
            )),
        replica_set_name='ext-prestable'),
    'ext': MongoReplicaSetTemplate(
        discovery=MongoDiscovery(
            static=[
                'sas2-5500-3d9-sas-qloud-mongo-26512.gencfg-c.yandex.net:26512',
                'man1-6702-man-qloud-mongo-26502.gencfg-c.yandex.net:26502',
                'vla2-3865-01c-vla-qloud-mongo-26522.gencfg-c.yandex.net:26522'],
            nanny_service=NannyMongoService(
                dcs=['man', 'sas', 'vla'],
                installation='external'
            )),
        replica_set_name='ext-root')
}


def get_nanny_instances(service_id, token):
    resp = requests.get(
        '{url}/v2/services/{service_id}/current_state/instances'.format(
            url=NANNY_URL,
            service_id=service_id),
        headers={
            'Authorization': 'OAuth %s' % token
        },
        verify=False
    )

    result = []
    if resp.status_code != 200:
        raise Exception('Unable to resolve instances for service %s' % service_id)
    body = resp.json()
    for res in body['result']:
        result.append('%s:%s' % (res['container_hostname'], res['port']))
    return result


def resolve_mongo_replica_set_addresses(service_description, token):
    result = set()
    for dc in service_description.dcs:
        instances = get_nanny_instances('%s_qloud_mongo_%s' % (
            dc, service_description.installation), token)
        result |= set(instances)
    return result


def connect(replica_set, user, password):
    client = MongoClient(replica_set.addresses,
                         replicaset=replica_set.replica_set_name,
                         username=user,
                         password=password,
                         authSource='admin',
                         authMechanism='SCRAM-SHA-1')
    return client


def connect_to_location(location_name, nanny_token, user, password):
    location = LOCATIONS[location_name]
    if nanny_token:
        addresses = resolve_mongo_replica_set_addresses(
            location.discovery.nanny_service, nanny_token)
    else:
        addresses = location.discovery.static
    return connect(
        MongoReplicaSet(
            addresses=addresses,
            replica_set_name=location.replica_set_name),
        user,
        password)
