import urllib
import urllib.request
import time
import threading
import json
import socket

try:
  import SimpleHTTPServer
except ImportError:
  import http.server as SimpleHTTPServer

try:
  from BaseHTTPServer import HTTPServer
except:
  from http.server import HTTPServer

from contextlib import closing
from signal import signal, SIGPIPE, SIG_DFL, SIGTERM, SIGINT


class ProgramKilled(Exception):
    pass


def signal_handler(signum, frame):
    raise ProgramKilled


signal(SIGPIPE, SIG_DFL)
signal(SIGTERM, signal_handler)
signal(SIGINT, signal_handler)


class StatCollector(threading.Thread):
    def __init__(self, address):
        threading.Thread.__init__(self)
        self.address = address
        self.stopped = threading.Event()
        self.rps = None
        self.avg_rt = None
        self.requests = None

    def run(self):
        while True:
            previous_requests = self.requests
            start_time = time.time()
            if self.stopped.wait(1):
                break

            self.requests, self.avg_rt = self._get_data()
            if previous_requests is not None:
                self.rps = (self.requests - previous_requests) / (time.time() - start_time)


    def _get_data(self):
        with closing(urllib.request.urlopen(self.address)) as resp:
            json_data = json.load(resp)
            avg_rt = max([w['avg_rt'] for w in json_data['workers']])
            requests = sum([w['requests'] for w in json_data['workers']])
            return requests, avg_rt

    def stop(self):
        self.stopped.set()


class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6


class HTTPRequestHandler(SimpleHTTPServer.SimpleHTTPRequestHandler):
    counter = None

    def do_GET(self):
        if self.path == '/ping':
            self.send_response(200)
            self.end_headers()
            self.wfile.write(b'pong')
            return

        self.send_response(200)
        self.end_headers()
        if self.counter:
            response = []
            rps = self.counter.rps
            if rps is not None:
                response.append('["uwsgi.rps_mxxx", %.2f]' % rps)
            avg_rt = self.counter.avg_rt
            if avg_rt is not None:
                response.append('["uwsgi.max_avg_rt_axxx", %.2f]' % avg_rt)
            self.wfile.write(b'[%s]' % (','.join(response)).encode())
        else:
            self.wfile.write(b'[]')

def main():
    from sys import argv
    target = argv[1]
    port = int(argv[2])

    counter = StatCollector(target)
    try:
        counter.start()
        HTTPRequestHandler.counter = counter
        server = HTTPServerV6(('', port), HTTPRequestHandler)
        keep_running = True
        server.serve_forever()
    except ProgramKilled:
        counter.stop()

if __name__ == "__main__":
    main()
