#!/usr/bin/python
# Provides: qloud_router_config
import json
import os
import re
import socket
import urllib2
from time import sleep

ETC_QLOUD_META_JSON = "/etc/qloud/meta.json"

QLOUD_NGINX_ROUTER_CONFIG = "/etc/default/qloud-nginx-router-config"

regex = r"set \$qloud_configuration_hash\s\'([a-z0-9]+)\'\;"


def get_config():
    qloud_router = "/etc/nginx/sites-enabled/qloud-router"
    if os.path.isfile(qloud_router):
        return qloud_router

    nginx_conf = "/etc/nginx/nginx.conf"
    if os.path.isfile(nginx_conf):
        return nginx_conf


def get_running_hash():
    return urllib2.urlopen("http://localhost/", timeout=3).read()


def get_conf_hash(config):
    for i in range(3):
        try:
            with open(config) as f:
                matches = re.finditer(regex, f.read(), re.MULTILINE)
                return matches.next().group(1)
        except Exception as _:
            sleep(3)


def load_variables():
    if os.path.isfile(ETC_QLOUD_META_JSON):
        with open(ETC_QLOUD_META_JSON) as f:
            data = json.load(f)
            config_url = data.get("user_environment", {}).get("QLOUD_ROUTER_CONFIG_URL", None)
            datacenter = data.get("datacenter", None)
            return config_url, datacenter


def load_nginx_loader_variables():
    if os.path.isfile(QLOUD_NGINX_ROUTER_CONFIG):
        with open(QLOUD_NGINX_ROUTER_CONFIG) as f:
            data = json.load(f)
            management_url = data.get("QLOUD_MANAGEMENT_URL")
            management_path = data.get("QLOUD_MANAGEMENT_PATH")
            return management_url, management_path


def check_qloud_hash(hash):
    headers = {"If-None-Match": '"' + hash + '"'}
    result = load_variables()
    try:
        if result is not None:
            config_url, datacenter = result
            if "/router/" in config_url:
                hostname = os.environ['PORTO_HOST']
                request = urllib2.Request(config_url + '/' + hostname, headers=headers)
            else:
                request = urllib2.Request(config_url + '/' + datacenter, headers=headers)
            request.get_method = lambda: 'HEAD'
            resp = urllib2.urlopen(request, timeout=10)
            check_responce(hash, resp)
            return
        result = load_nginx_loader_variables()
        if result is not None:
            hostname = socket.getfqdn()
            url, path = result
            request = urllib2.Request(url + '/' + path + '/config/' + hostname, headers=headers)
            request.get_method = lambda: 'HEAD'
            resp = urllib2.urlopen(request, timeout=10)
            check_responce(hash, resp)
            return
    except urllib2.HTTPError as err:
        if err.code != 304:
            raise err


def check_responce(hash, resp):
    if "etag" in resp.headers:
        raise Exception("Conf file hash does not match with qloud-manager hash; qloud_hash " + resp.headers["etag"]
                        + ", file_hash " + hash)
    else:
        raise Exception("Cannot get router config from qloud-manager")


def message(code, m):
    name = os.path.basename(__file__).split('.')[0]
    msg = "PASSIVE-CHECK:" + name + ";" + str(code) + ";" + m
    print(msg)


if __name__ == '__main__':
    try:
        config = get_config()
        if config is None:
            raise Exception("Main config file not found")

        config_hash = get_conf_hash(config)
        if config_hash is None:
            raise Exception("Do not get hash after 10 times with 3 sec sleep")

        check_qloud_hash(config_hash)

        running_hash = get_running_hash()
        if running_hash != config_hash:
            raise Exception(
                "Inconsist qloud-nginx running config; file_hash " + config_hash + ", runnning_hash " + running_hash)
        message(0, "Actual config of qloud-nginx is running")
    except Exception as e:
        message(2, str(e))
