import os

import requests as rq


class EnvironmentNotFound(Exception):
    def __init__(self, environment):
        super(EnvironmentNotFound, self).__init__('Environment {} not found'.format(environment))


class QloudError(Exception):
    def __init__(self, code, message):
        super(QloudError, self).__init__('Qloud respond with {}: {}'.format(code, message))


class AuthorizationError(Exception):
    def __init__(self, url):
        super(AuthorizationError, self).__init__('Not authorized to access {}'.format(url))


class Qloud:
    hostname = None

    def __init__(self, hostname):
        self.hostname = hostname

    def dump_environment(self, id):
        url = 'https://{}/api/v1/environment/dump/{}'.format(self.hostname, id)
        return self.handle_response(rq.get(url, headers=self.__headers()), {
            200: lambda r: r.json(),
            403: lambda r: AuthorizationError(url),
            404: lambda r: EnvironmentNotFound(id),
            None: lambda r: QloudError(r.status_code, r.text)
        })

    def get_environment_status(self, id, version=None):
        if version is not None:
            url = 'https://{}/api/v1/environment/status/{}/{}'.format(self.hostname, id, version)
        else:
            url = 'https://{}/api/v1/environment/status/{}'.format(self.hostname, id)

        return self.handle_response(rq.get(url, headers=self.__headers()), {
            200: lambda r: r.json(),
            403: lambda r: AuthorizationError(url),
            404: lambda r: EnvironmentNotFound(id),
            None: lambda r: QloudError(r.status_code, r.text)
        })

    def upload(self, id, data):
        data['objectId'] = id
        url = 'https://{}/api/v1/environment/upload'.format(self.hostname)
        return self.handle_response(
            rq.post(url, headers=self.__headers({'Content-Type': 'application/json'}), json=data),
            {
                200: lambda r: r.json(),
                403: lambda r: AuthorizationError(url),
                None: lambda r: QloudError(r.status_code, r.text)
            }
        )

    @staticmethod
    def handle_response(resp, actions):
        code = resp.status_code
        action = code

        if code not in actions.keys():
            if None in actions.keys():
                action = None
            else:
                raise Exception('Remote server respond with unhandled code {}: {}'.format(code, resp.text))

        result = actions[action](resp)
        if isinstance(result, Exception):
            raise result

        return result

    @staticmethod
    def __headers(h=None):
        result = h
        if not result:
            result = {}
        result['Authorization'] = 'OAuth {}'.format(os.getenv('OAUTH_TOKEN'))
        return result
