import logging
import os

LOG = logging.getLogger(__name__)
DEFAULT_CONFIG_MODULE = os.getenv("QNOTIFIER_CONFIG_MODULE")


def make_flask_app(config_module_name=None):
    import flask
    from flask_migrate import Migrate
    from werkzeug.middleware.proxy_fix import ProxyFix
    from werkzeug.contrib.fixers import HeaderRewriterFix
    import alembic.config as alconfig
    import alembic.command as alcommand
    from lib import api, context
    from lib.utils.flask_masterslave import MasterSlaveSqlAlchemy, FlaskMasterSlave, MasterSlaveConnectionPool
    from lib.utils.flask_yauth import Yauth

    class RelEnvironment(flask.templating.Environment):
        def join_path(self, template, parent):
            return os.path.join(os.path.dirname(parent), template)

    class Flask(flask.Flask):
        jinja_environment = RelEnvironment

    config_module_name = config_module_name or DEFAULT_CONFIG_MODULE
    LOG.info("default config name: %r", DEFAULT_CONFIG_MODULE)
    LOG.info("config_module_name = %r (is default: %r)", config_module_name, config_module_name == DEFAULT_CONFIG_MODULE)
    app = Flask(__name__)
    app.wsgi_app = ProxyFix(app.wsgi_app, x_proto=1)

    app.config.from_object('app.config')
    if config_module_name is not None:
        app.config.from_object(config_module_name)

    app.wsgi_app = HeaderRewriterFix(
        app.wsgi_app,
        add_headers=(
            ('Access-Control-Allow-Origin', '*'),
            ('Access-Control-Allow-Methods', 'GET, POST, DELETE, PUT, PATCH, OPTIONS'),
            ('Access-Control-Allow-Headers', 'access-control-allow-origin, Content-Type, Authorization'),
            ('X-Served-By', app.config['FQDN']),
        )
    )

    @app.route('/ping')
    def ping():
        return 'OK'

    with app.app_context():
        app.database = MasterSlaveSqlAlchemy(app)
        Migrate(app, app.database)

        FlaskMasterSlave(app)
        Yauth(app)

        alembic_cfg = alconfig.Config('migrations/alembic.ini')
        with app.database.engine.begin() as connection:
            alembic_cfg.attributes['connection'] = connection
            alembic_cfg.set_main_option('script_location', 'migrations')
            alcommand.upgrade(alembic_cfg, 'head')

        app.database_pool = MasterSlaveConnectionPool(app)
        app.api_context = context.Context(log=logging.getLogger('qn.ctx'))
        app.register_blueprint(api.base)

    return app
