import os
import sys
import logging
import logging.config
import argparse
from collections import defaultdict

from concurrent import futures

import psycopg2
import psycopg2.pool

LOG = logging.getLogger(__name__)
DEFAULT_CONFIG = os.getenv("QNOTIFIER_CONFIG_MODULE")


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--debug', action='store_true', default=False, help='Debug flask app')
    parser.add_argument('--cfg_name', default=DEFAULT_CONFIG, help='config module name')
    parser.add_argument('--log_dir', default=None, help='logs directory')
    return parser.parse_args()


def make_flask_app(cfgmodule=None):
    import flask
    from werkzeug.middleware.proxy_fix import ProxyFix

    cfgmodule = cfgmodule or DEFAULT_CONFIG
    app = flask.Flask(__name__)
    app.wsgi_app = ProxyFix(app.wsgi_app)
    app.config.from_object('app.config_worker')
    if cfgmodule is not None:
        app.config.from_object(cfgmodule)

    with app.app_context():
        app.database = psycopg2.pool.ThreadedConnectionPool(
            minconn=1,
            maxconn=2,
            dsn=app.config['SQLALCHEMY_DATABASE_URI'],
        )
        app.unistats = defaultdict(int)

    @app.route('/ping')
    def ping():
        return 'OK'

    @app.route('/unistat')
    def unistat():
        return flask.jsonify(list(app.unistats.items()))

    return app


def main():
    args = parse_args()

    app = make_flask_app(args.cfg_name)
    logging.config.dictConfig(app.config['LOGGING'])

    with app.app_context():
        from lib.notifiers.worker import Worker
        from lib.logbroker import PQClient

        # TODO add option to specify amount of workers
        pq = PQClient(
            pq_config=app.config['PQ']['server'],
            reader_config=app.config['PQ']['reader'],
            writer_config=app.config['PQ']['writer'],
        )
        pq.start().result(timeout=60)
        thr = Worker(app.config['WORKER'], pq, unistats=app.unistats)
        thr.start()

        try:
            app.run(
                host=app.config['WSGI_LISTEN_ADDRESS'][0],
                port=app.config['WSGI_LISTEN_ADDRESS'][1],
                debug=args.debug,
                use_reloader=False,
                threaded=True
            )
        except Exception:
            logging.getLogger('qnotifier').exception("app run failed:")
        finally:
            stop = thr.shutdown()
            futures.wait([stop], timeout=300, return_when=futures.ALL_COMPLETED)


if __name__ == '__main__':
    sys.path.insert(0, os.getcwd())
    main()
