import os
import sys
import logging
import logging.config
import argparse
from collections import defaultdict

import psycopg2
import psycopg2.pool


LOG = logging.getLogger(__name__)
DEFAULT_CONFIG = os.getenv("QNOTIFIER_CONFIG_MODULE")


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--debug', action='store_true', default=False, help='Debug flask app')
    parser.add_argument('--cfg_name', default=DEFAULT_CONFIG, help='config module name')
    parser.add_argument('--log_dir', default=None, help='logs directory')
    return parser.parse_args()


def make_flask_app(cfgmodule=None):
    import flask

    cfgmodule = cfgmodule or DEFAULT_CONFIG
    app = flask.Flask(__name__)
    app.config.from_object('app.config_telegram')
    if cfgmodule is not None:
        app.config.from_object(cfgmodule)

    with app.app_context():
        app.database = psycopg2.pool.ThreadedConnectionPool(
            minconn=1,
            maxconn=2,
            dsn=app.config['SQLALCHEMY_DATABASE_URI'],
        )
        app.unistats = defaultdict(int)

    @app.route('/ping')
    def ping():
        return 'OK'

    @app.route('/unistat')
    def unistat():
        return flask.jsonify(list(app.unistats.items()))

    return app


def main():
    args = parse_args()

    app = make_flask_app(args.cfg_name)
    logging.config.dictConfig(app.config['LOGGING'])

    with app.app_context():
        from lib.telegram_bot import TelegramBot
        from lib.exclusiveservice import ExclusiveService

        bot = TelegramBot(app.config['BOT'], unistats=app.unistats)
        lock = ExclusiveService(
            cfg=app.config['BOT']['lock'],
            runnable=bot.start,
            name='qnotifier_telegram_bot_' + ('test' if app.config['TESTING'] else 'prod'),
        )
        lock.start()

        try:
            app.run(
                host=app.config['WSGI_LISTEN_ADDRESS'][0],
                port=app.config['WSGI_LISTEN_ADDRESS'][1],
                debug=args.debug,
                use_reloader=False,
                threaded=True
            )
        except Exception:
            logging.getLogger('qnotifier').exception("app run failed:")
        finally:
            bot.shutdown()


if __name__ == '__main__':
    sys.path.insert(0, os.getcwd())
    main()
