import logging
import re

import flask
from flask_restplus import Api, Resource, fields, marshal

from .utils.flask_masterslave import db_readonly
from .utils.flask_yauth import (
    yauth_required, check_permissions_for,
    AuthenticationError, NotAuthorizedError, YaPermissionError
)
from .utils.dbutil import ConnectionErrorFor503

LOG = logging.getLogger(__name__)


authorizations = {
    'tvm': {
        'type': 'apiKey',
        'in': 'header',
        'name': 'X-Ya-Service-Ticket',
    },
    'oauth': {
        'type': 'oauth2',
        'flow': 'implicit',
        'authorizationUrl': 'https://oauth.yandex-team.ru/authorize?response_type=code',
        'scopes': {},
    },
    # 'Session_id': {
    #     'type': 'apiKey',
    #     'in': 'cookie',
    #     'name': 'Session_id',
    # },
    # 'sessionid2': {
    #     'type': 'apiKey',
    #     'in': 'cookie',
    #     'name': 'sessionid2'
    # },
}
user_or_service = [
    'tvm',
    {'oauth': []},
    # {'Session_id': {}, 'sessionid2': {}},
]

# base to dispatch statics and templates for ui
base = flask.Blueprint('qnotifier_doc',
                       __name__,
                       template_folder='../templates',
                       static_folder='../static',
                       static_url_path='/ui',
                       )


class TagsParseError(Exception):
    pass


api = Api(base,
          version='1.0',
          default='events',  # see https://github.com/noirbizarre/flask-restplus/issues/460
          default_label='Event operations',
          title='API',
          description='Qloud events API',
          authorizations=authorizations,
          )


# defined after Api to override 'swagger_static' template defined by flask-restplus
@base.record_once
def init_static(state):
    @state.app.template_global()
    def swagger_static(filename):
        return flask.url_for('qnotifier_doc.static', filename=filename)

    globals()['swagger_static'] = swagger_static


@api.documentation
def api_ui():
    return flask.render_template('ui/swagger-ui.html', title=api.title, specs_url=api.specs_url)


events_ns = api.namespace('events', description='Event operations')
subscriptions_ns = api.namespace('subscriptions', description='Subscribe operations')
settings_ns = api.namespace('settings', description='Settings operations')
search_ns = api.namespace('search', description='Search operations')
telegram_ns = api.namespace('telegram', description='Telegram operations')

error_model = api.model('Error', {
    'error': fields.String(required=True, description="Error type"),
    'message': fields.String(description="Error message"),
})

subscription_model = subscriptions_ns.model('Subscription', {
    'tags': fields.List(fields.String,
                        description="Tags in mask. Tags with prefix '!' are excluded"),
    'options': fields.Raw(default={}),
})
subscriptions_model = subscriptions_ns.model('Subscriptions', {
    'subscriptions': fields.List(fields.Nested(subscription_model)),
    'options': fields.Raw(default={}),
})
name_subscription_model = api.model('NameSubscription', {
    'name': fields.String(required=True, description='User or group subscribed to the subscription'),
    'subscription': fields.Nested(subscription_model),
})
matching_subscriptions_model = subscriptions_ns.model('MatchingSubscriptions', {
    'subscriptions': fields.List(fields.Nested(name_subscription_model)),
})

event_model = events_ns.model('Event', {
    'message': fields.String(required=True, description='Event message'),
    'tags': fields.List(fields.String, required=True,
                        description="Tags of event"),
    'forced-users': fields.List(
        fields.String, required=False,
        description="Users and/or groups that should be notified even if they have no subscription"
    ),
    'extra': fields.Raw(description='Any extra options in key-value format', required=False),
})

settings_model = settings_ns.model('Settings', {})


def make_exc_handler(code, *exc_classes):
    @api.marshal_with(error_model)
    def _wrapper(error):
        logging.getLogger('qnotifier.api').exception("request failed: %s", error)
        return {
                   'error': type(error).__name__,
                   'message': str(error),
               }, code

    for exc_class in exc_classes:
        _wrapper = api.errorhandler(exc_class)(_wrapper)
    return _wrapper


_auth_failed = make_exc_handler(403, AuthenticationError)
_not_authorized = make_exc_handler(401, NotAuthorizedError)
_tags_parse_error = make_exc_handler(400, TagsParseError)
_permission_error = make_exc_handler(403, YaPermissionError)
_database_connection_error = make_exc_handler(503, ConnectionErrorFor503)
_unhandled_exception = make_exc_handler(500, Exception)


@yauth_required(service=True)
@events_ns.route('/')
class Event(Resource):
    @db_readonly
    @api.expect(event_model)
    @api.doc('Push event to affected users', security=['tvm'])
    @api.response(403, 'TVM service ticket is invalid', model=error_model)
    @api.response(401, 'No TVM service ticket provided', model=error_model)
    @api.response(400, 'Incorrect request', model=error_model)
    @api.response(204, "Event processed")
    def post(self):
        LOG.debug("Event.post")

        ctx = flask.current_app.api_context
        event = ctx.parse_event(events_ns.payload)
        ctx.dispatch_event(event)
        return create_204()


@search_ns.route('/by-tags')
class SearchByTags(Resource):
    @api.doc('Find all subscriptions, matchin specified tags set',
             params={
                 'exact': 'If this flag is set to true, only subscriptions with exactly '
                          'matching tags will be returned, otherwise all the '
                          'subscriptions where specified set is superset will be found',
             })
    @api.param('tags', explode=True, type=[str], collectionFormat='multi',
               description='subscriptions selector')
    @api.response(200, 'Subscriptions found', model=matching_subscriptions_model)
    @api.response(400, 'No tags provided', model=error_model)
    def get(self):
        parser = search_ns.parser()
        parser.add_argument('tags', action='append', location='args')
        parser.add_argument('exact', location='args', default='')
        args = parser.parse_args()
        tags = args['tags']
        exact = args['exact'].lower() in ['true', 'yes', 'da', '1']

        LOG.debug('SearchByTags.get: exact=%s, tags=%s', exact, tags)
        if not tags:
            raise TagsParseError("No tags given")

        mask = parse_mask(tags)

        users, groups = flask.current_app.api_context.find_subscriptions_by_tags(tags, exact)
        result = []
        for user, subs in users.items():
            for sub in subs.values():
                result.append({'name': user, 'subscription': sub})
        for group, subs in groups.items():
            for sub in subs.values():
                result.append({'name': GROUP_PREFIX + group, 'subscription': sub})

        return marshal({
            'subscriptions': result,
        }, matching_subscriptions_model)


@subscriptions_ns.route('/<name>')
class Subscriptions(Resource):
    @api.doc('Get user or group subscriptions',
             params={
                 'name': "User name or group name in form 'group:name'",
             })
    @api.param('tags', explode=True, type=[str], collectionFormat='multi',
               # example=['priority:urgent', 'service:myservice'],
               description='optional subscription selector')
    @api.response(200, "Subscriptions found", model=subscriptions_model)
    def get(self, name):
        LOG.debug("Subscriptions.get: name=%r", name)

        is_group, name = parse_name(name)
        LOG.debug("is_group=%r, name=%r", is_group, name)

        parser = subscriptions_ns.parser()
        parser.add_argument('tags', action='append', location='args')
        tags = parser.parse_args()['tags']
        LOG.debug("tags: %s", tags)
        try:
            mask = parse_mask(tags)
        except TagsParseError:
            LOG.debug("TagsParseError")
            mask = None

        LOG.debug("call get_subscriptions")
        subscriptions = flask.current_app.api_context.get_subscriptions(name, is_group)
        LOG.debug("subscriptions [1]: %s", subscriptions)

        if mask:
            mask = set(mask)
            subscriptions = list(filter(lambda i: set(i[0]) == mask, subscriptions))
            LOG.debug("subscriptions [2]: %s", subscriptions)

        global_settings = flask.current_app.api_context.get_settings(name, is_group)
        LOG.debug("global_settings: %r", global_settings)

        if not subscriptions and not global_settings:
            # error_message = "%s %r subscriptions not found" % ("Group" if is_group else "User", name)
            subscriptions = dict()
            global_settings = dict()

        return marshal({
            'subscriptions': [dict(tags=sub, options=options) for sub, options in subscriptions],
            'options': global_settings,
        }, subscriptions_model)

    @yauth_required(service=True, user=True)
    @api.response(403, 'Authentication info is invalid or no access to target user or group', model=error_model)
    @api.response(401, 'No authentication info provided', model=error_model)
    @api.response(400, "Invalid request", model=error_model)
    @api.response(204, "Subscription created")
    @api.expect(subscription_model, validate=True)
    @api.doc('Add user or group subscription',
             params={"name": "User name or group name in form 'group:name'"},
             security=user_or_service,
             )
    def post(self, name):
        LOG.debug("Subscriptions.post: name=%r", name)

        mask = parse_mask(subscriptions_ns.payload['tags'])

        is_group, name = parse_name(name)
        check_permissions_for(name, is_group)

        flask.current_app.api_context.add_subscription(
            name,
            is_group,
            mask,
            subscriptions_ns.payload.get('options') or {}
        )

        return create_204()

    @yauth_required(service=True, user=True)
    @api.response(404, "Subscription or user is not found", model=error_model)
    @api.response(403, 'Authentication info is invalid or no access to target user or group', model=error_model)
    @api.response(401, 'No authentication info provided', model=error_model)
    @api.response(400, "Invalid request", model=error_model)
    @api.response(204, "Subscription updated")
    @api.expect(subscription_model, validate=True)
    @api.doc('Update user or group subscription settings',
             params={'name': "User name or group name in form 'group:name'"},
             security=user_or_service,
             )
    def patch(self, name):
        LOG.debug("Subscriptions.patch: name=%r", name)

        mask = parse_mask(subscriptions_ns.payload['tags'])

        if not subscriptions_ns.payload['options']:
            return marshal({"error": "options cannot be empty"}, error_model), 400

        is_group, name = parse_name(name)
        check_permissions_for(name, is_group)

        try:
            flask.current_app.api_context.update_subscription(
                name,
                is_group,
                mask,
                subscriptions_ns.payload['options']
            )
        except ValueError as e:
            return marshal({"error": e.args[0]}, error_model), 402

        return create_204()

    @yauth_required(service=True, user=True)
    @api.response(404, "Subscription or user is not found", model=error_model)
    @api.response(403, 'Authentication info is invalid or no access to target user or group', model=error_model)
    @api.response(401, 'No authentication info provided', model=error_model)
    @api.response(400, "Invalid request", model=error_model)
    @api.response(204, "Subscription removed")
    @api.expect(subscription_model, validate=True)
    @api.doc('Remove user or group subscription',
             params={'name': "User name or group name in form 'group:name'"},
             security=user_or_service,
             )
    def delete(self, name):
        LOG.debug("Subscriptions.delete: name=%r", name)

        mask = parse_mask(subscriptions_ns.payload['tags'])

        is_group, name = parse_name(name)
        check_permissions_for(name, is_group)

        try:
            flask.current_app.api_context.remove_subscription(name, is_group, mask)
        except KeyError as e:
            return marshal({"error": e.args[0]}, error_model), 404

        return create_204()


@settings_ns.route('/<name>')
class Settings(Resource):
    @api.response(404, 'Settings not found', model=error_model)
    @api.response(200, 'Settings found')
    @api.doc('Get user or group settings',
             params={'name': "User name or group name in form 'group:name'"})
    def get(self, name):
        LOG.debug("Settings.get: name=%r", name)

        is_group, name = parse_name(name)
        settings = flask.current_app.api_context.get_settings(name, is_group)
        if not settings:
            return marshal({"error": "settings not found"}, error_model), 400

        return settings

    @yauth_required(service=True, user=True)
    @api.expect(settings_model, validate=True)
    @api.response(403, 'Authentication info is invalid or no access to target user or group', model=error_model)
    @api.response(401, 'No authentication info provided', model=error_model)
    @api.response(204, "Settings stored")
    @api.doc("Store user or group settings",
             params={'name': "User name or group name in form 'group:name'"},
             security=user_or_service,
             )
    @api.response(204, 'Settings stored')
    def post(self, name):
        LOG.debug("Settings.post: name=%r", name)

        is_group, name = parse_name(name)
        check_permissions_for(name, is_group)

        try:
            flask.current_app.api_context.update_settings(name, is_group, settings_ns.payload, replace_all=True)
        except Exception as e:
            return marshal({"error": str(e)}, error_model), 500

        return create_204()

    @yauth_required(service=True, user=True)
    @api.response(403, 'Authentication info is invalid or no access to target user or group', model=error_model)
    @api.response(401, 'No authentication info provided', model=error_model)
    @api.response(204, "Settings updated")
    @api.expect(settings_model, validate=True)
    @api.doc("Update user or group settings",
             params={'name': "User name or group name in form 'group:name'"},
             security=user_or_service,
             )
    def patch(self, name):
        LOG.debug("Settings.patch: name=%r", name)

        is_group, name = parse_name(name)
        check_permissions_for(name, is_group)

        try:
            flask.current_app.api_context.update_settings(name, is_group, settings_ns.payload, replace_all=False)
        except Exception as e:
            return marshal({"error": str(e)}, error_model), 500

        return create_204()


@telegram_ns.route('/<name>/check')
class TelegramCheck(Resource):
    @api.response(200, 'Result found', model=telegram_ns.model('TelegramCheckResult', {'authorized': fields.Boolean()}))
    @api.response(400, 'Group check attempted', model=error_model)
    @api.doc('Check if user is authorized within telegram bot',
             params={'name': "User name"})
    def get(self, name):
        LOG.debug("TelegramCheck.get: name=%r", name)

        is_group, name = parse_name(name)
        if is_group:
            return marshal({"error": "groups are not supported"}, error_model), 400

        result = flask.current_app.api_context.check_telegram_authentication(name)

        return {'authorized': result}, 200


GROUP_PREFIX = "group:"


def parse_name(name):
    is_group = name.startswith(GROUP_PREFIX)
    if is_group:
        name = name[len(GROUP_PREFIX):]

    return is_group, name


TAG_REGEXP = re.compile(r"^\s*(!?)\s*(\S+)\s*$", re.DOTALL)


def parse_tag(tag):
    if '&' in tag:
        raise TagsParseError("Tag cannot contain '&' char: %r" % (tag,))

    match = TAG_REGEXP.match(tag)
    if not match:
        raise TagsParseError("Tag %r does not match %r" % (tag, TAG_REGEXP.pattern))

    prefix, body = match.group(1, 2)
    return prefix + body


def parse_mask(mask):
    LOG.debug("=> parse_mask: %r", mask)
    if not isinstance(mask, list) or not all(isinstance(tag, str) for tag in mask):
        LOG.debug("Request cannot be parsed (list of strings expected)")
        raise TagsParseError("Request cannot be parsed (list of strings expected)")

    if not mask:
        LOG.debug("Tag list cannot be empty")
        raise TagsParseError("Tag list cannot be empty")

    # TODO: write tests for tag format
    return [parse_tag(v) for v in mask]


def create_204():
    # NOTE: it's workaround for non-zero content-length bug in werkzeug 0.14.1
    # see https://github.com/flask-restful/flask-restful/issues/736
    resp = flask.make_response('', 204)
    resp.headers['Content-Length'] = 0
    return resp
