from itertools import chain
import logging

from . import events, users, groups, subscriptions, queues

import flask


class Context(object):
    def __init__(self, log=None):
        self.event_sources = []
        self.subscriptions_source = None
        self.user_sources = []
        self.group_sources = []
        self.queue = None
        self.notifiers = []
        self.log = log or logging.getLogger('ctx')

    def init(self):
        """
        :raises TypeError: if config contains source of unknown type
        :raises ValueError: if source cannot be created with specified params
        """

        self.notifiers = flask.current_app.config['NOTIFIERS']
        self._set_subscriptions_source()
        self._set_event_sources()
        self._set_user_sources()
        self._set_group_sources()
        self._set_queue()

    def _set_event_sources(self):
        for item in flask.current_app.config['EVENTS']:
            self.log.info("initializing: %s", item)
            klass = item['classname']
            params = item['options']
            source_type = getattr(events, klass, None)
            if source_type is None:
                raise TypeError("events config has invalid class: %r" % (klass,))

            try:
                source = source_type.create(params)
            except Exception as e:
                raise ValueError("event source %r cannot be created: %s" % (klass, e))
            else:
                self.log.info("initialized: %s", source)
                self.event_sources.append(source)

    def _set_user_sources(self):
        for item in flask.current_app.config['USERS']:
            self.log.info("initializing: %s", item)
            klass = item['classname']
            params = item['options']
            source_type = getattr(users, klass, None)
            if source_type is None:
                raise TypeError("users config has invalid class: %r" % (klass,))

            try:
                source = source_type.create(params, self.log)
            except Exception as e:
                raise ValueError("user source %r cannot be created: %s" % (klass, e))
            else:
                self.log.info("initialized: %s", source)
                self.user_sources.append(source)

    def _set_group_sources(self):
        for item in flask.current_app.config['GROUPS']:
            self.log.info("initializing: %s", item)
            klass = item['classname']
            params = item['options']
            source_type = getattr(groups, klass, None)
            if source_type is None:
                raise TypeError("groups config has invalid class: %r" % (klass,))

            try:
                source = source_type.create(params, self.log)
            except Exception as e:
                raise ValueError("group source %r cannot be created: %s" % (klass, e))
            else:
                self.log.info("initialized: %s", source)
                self.group_sources.append(source)

    def _set_subscriptions_source(self):
        self.log.info("initializing: %s", flask.current_app.config['SUBSCRIPTIONS'])
        klass = flask.current_app.config['SUBSCRIPTIONS']['classname']
        params = flask.current_app.config['SUBSCRIPTIONS']['options']
        source_type = getattr(subscriptions, klass, None)
        if source_type is None:
            raise TypeError("subscriptions config has invalid class: %r" % (klass,))

        try:
            source = source_type.create(params, self.log)
        except Exception as e:
            raise ValueError("subscription source %r cannot be created: %s" % (klass, e))
        else:
            self.log.info("initialized: %s", source)
            self.subscriptions_source = source

    def _set_queue(self):
        self.log.info("initializing: %s", flask.current_app.config['QUEUE'])
        klass = flask.current_app.config['QUEUE']['classname']
        params = flask.current_app.config['QUEUE']['options']
        queue_type = getattr(queues, klass, None)
        if queue_type is None:
            raise TypeError("queue config has invalid class: %r" % (klass,))

        try:
            queue = queue_type.create(params, self.log)
        except Exception as e:
            raise ValueError("queue %r cannot be created: %s" % (klass, e))
        else:
            self.log.info("initialized: %s", queue)
            self.queue = queue

    def _find_user(self, login):
        for source in self.user_sources:
            user = source.find_user(login)
            if user is not None:
                return user
        return None

    def _resolve_group(self, name):
        for source in self.group_sources:
            users = source.find_users(name)
            if users:
                return users
        return set()

    def dispatch_event(self, event):
        message, params = event.source.make_message(event)  # FIXME
        affected_users, affected_groups = self.subscriptions_source.subscribers_for(event.tags)
        for user in event.forced_users:
            affected_users.setdefault(user, {})

        for group, group_options in affected_groups.items():
            logins = self._resolve_group(group)
            for login in logins:
                affected_users.setdefault(login, {})
                for k, v in group_options.items():
                    affected_users[login].setdefault(k, v)

        users = {
            login: self._find_user(login)
            for login in affected_users
        }

        for notifier in self.notifiers:
            enabled_logins = [
                login
                for login in affected_users
                if users[login] is not None
                and affected_users[login].get((notifier, 'enabled'), "true").lower() != "false"
            ]
            self.log.info("event tags %s, forced users %s, via %s will dispatch to: %s", event.tags, event.forced_users, notifier, enabled_logins)
            for login in enabled_logins:
                # TODO use batch, let notifier choose, should 'em be sent one-by-one or not
                address = users[login].get_address_for(notifier)
                if address is not None:
                    self.queue.put({
                        "type": notifier,
                        "address": address,
                        "attempt": 0,
                        "message": message,
                        "options": params,
                    })

    def get_subscriptions(self, name, is_group=False):
        if is_group:
            result = self.subscriptions_source.subscriptions_for_group(name)
        else:
            result = self.subscriptions_source.subscriptions_for_user(name)
        return result or []

    def get_settings(self, name, is_group=False):
        s = self.subscriptions_source.settings_for(name, is_group=is_group)
        return s or {}

    def update_settings(self, name, is_group, settings, replace_all):
        self.subscriptions_source.update_settings_for(name, is_group, settings, replace_all)

    def add_subscription(self, login, is_group, tags, settings):
        return self.subscriptions_source.add_subscription(login, is_group, tags, settings)

    def update_subscription(self, login, is_group, tags, settings):
        return self.subscriptions_source.update_subscription(login, is_group, tags, settings)

    def remove_subscription(self, login, is_group, tags):
        return self.subscriptions_source.remove_subscription(login, is_group, tags)

    def find_subscriptions_by_tags(self, tags, exact):
        return self.subscriptions_source.find_subscriptions_by_tags(tags, exact)

    def parse_event(self, data):
        for source in self.event_sources:
            event = source.parse_event(data)
            if event is not None:
                return event

        raise ValueError("Event cannot be parsed")

    def check_telegram_authentication(self, name):
        user = self._find_user(name)
        if user is None:
            return False

        tg = user.get_address_for('telegram')
        if tg is None:
            return False

        for source in self.user_sources:
            try:
                return source.check_telegram_authentication(tg['nickname'])
            except NotImplementedError:
                pass

        return False

    def shutdown(self):
        for source in chain(
            (self.subscriptions_source,),
            self.event_sources,
            self.user_sources,
            self.group_sources,
            (self.queue,),
        ):
            source.stop()

    def __del__(self):
        self.shutdown()
