import os
import enum
import time
import random
import logging
import threading

import ylock


class RestartPolicy(enum.Enum):
    DO_NOT_RESTART = 1
    RESTART_ON_EXCEPTION = 2
    ALWAYS_RESTART = 3


class RetryFailedError(Exception):
    """
    Raised when retrying an operation ultimately failed, after
    retrying the maximum number of attempts.
    """
    pass


class RetrySleeper(object):
    """
    A retry sleeper that will track its jitter, backoff and
    sleep appropriately when asked.
    """
    def __init__(self, max_tries=-1, delay=0.5, backoff=2, max_jitter=0.8,
                 max_delay=600):
        """
        Create a :class:`RetrySleeper` instance

        :param max_tries: How many times to retry the command. -1 for unlimited.
        :param delay: Initial delay between retry attempts.
        :param backoff: Backoff multiplier between retry attempts.
                        Defaults to 2 for exponential backoff.
        :param max_jitter: Additional max jitter period to wait between
                           retry attempts to avoid slamming the server.
        :param max_delay: Maximum delay in seconds, regardless of other
                          backoff settings. Defaults to ten minutes.

        """
        self.max_tries = max_tries
        self.delay = delay
        self.backoff = backoff
        self.max_jitter = max_jitter
        self.max_delay = float(max_delay)
        self._attempts = 0
        self._cur_delay = delay

    @property
    def cur_delay(self):
        return self._cur_delay

    def reset(self):
        """
        Reset the attempt counter
        """
        self._attempts = 0
        self._cur_delay = self.delay

    def increment(self, exception=True):
        """
        Increment the failed count, and sleep appropriately before
        continuing
        :return: False if max attempts reached, True otherwise
        :rtype: bool
        """
        try:
            time_to_sleep = self.get_next_time_to_sleep()
        except RetryFailedError:
            if exception:
                raise
            return False
        else:
            time.sleep(time_to_sleep)
            return True

    def get_next_time_to_sleep(self):
        """
        Increment the failed count and just return delay before the next retry
        and do not sleep
        :rtype: float
        """
        if self._attempts == self.max_tries:
            raise RetryFailedError("Too many retry attempts")
        self._attempts += 1
        jitter = random.random() * self.max_jitter
        result = min(self._cur_delay + jitter, self.max_delay)
        self._cur_delay = min(self._cur_delay * self.backoff, self.max_delay)
        return result


class ExclusiveService(threading.Thread):
    DEFAULT_ACQUIRE_TIMEOUT_STRATEGY = lambda: None

    def __init__(
        self, cfg, name, runnable,
        acquire_timeout_strategy=DEFAULT_ACQUIRE_TIMEOUT_STRATEGY,
        restart_policy=RestartPolicy.DO_NOT_RESTART
    ):
        super().__init__()
        self.daemon = True

        self._service = None
        if callable(runnable):
            self._service_target = runnable
        else:
            self._service_target = runnable.run

        self.service_name = name
        self.manager = ylock.create_manager(**cfg)
        self.log = logging.getLogger(f'exclusive.{name}')

        self._acquire_timeout_strategy = acquire_timeout_strategy
        self._restart_policy = restart_policy

    def run(self) -> None:
        sleeper = RetrySleeper(max_delay=5)
        lock = self.manager.lock(name=self.service_name, block=True)
        timeout_supported = 'YT' in lock.acquire.__func__.__qualname__  # NOTE (torkve) zomg

        while True:
            acquire_timeout = self._acquire_timeout_strategy()
            self.log.info(f"acquiring lock with timeout of {acquire_timeout} seconds...")

            args = dict(timeout=acquire_timeout) if timeout_supported else {}

            try:
                acquired = lock.acquire(**args)
            except Exception as e:
                self.log.info(f'failed to acquire lock: {e}')
                lock.release()
                sleeper.increment()
                continue

            if not acquired:
                self.log.info(f'failed to acquire lock within timeout of {acquire_timeout} seconds')
                continue

            try:
                # we locked, yey!
                # now start service and wait for session state change
                sleeper.reset()
                self.log.info("became singleton - starting service")

                # here in qnotifier it's just thread start function, so it evaluates and exits
                self._service_target()
                while True:
                    time.sleep(10)
                    if not lock.check_acquired():
                        self.log.info("Lock has been lost, we'd better die")
                        os._exit(0)
            except Exception:
                lock.release()
                self.log.info("service raised an exception")
                if self._restart_policy not in (RestartPolicy.ALWAYS_RESTART, RestartPolicy.RESTART_ON_EXCEPTION):
                    break
                self.log.info("restarting according to the restart policy...")
            else:
                self.log.info("service finished")
                if self._restart_policy != RestartPolicy.ALWAYS_RESTART:
                    break
                self.log.info("restarting according to the restart policy...")

            # sleep for sometime to let someone take leadership
            time.sleep(2)
