import logging
import requests
from datetime import datetime

from .abstract import Group, GroupSource


class AbcGroups(GroupSource):
    API_SERVICES_URL = "https://abc-back.yandex-team.ru/api/v4/services/?state__in=develop,supported,needinfo&fields=slug,ancestors.slug,id&page_size=5000"
    API_MEMBERS_URL = "https://abc-back.yandex-team.ru/api/v4/services/members/?service__state__in=develop,supported,needinfo&state=approved&fields=service.id,service.slug,role.scope.slug,person.login&page_size=20000"

    def __init__(self, oauth, asynchronously=False, log=None):
        self.oauth = oauth
        self.groups = {}
        self.log = log or logging.getLogger('abc')
        # TODO spawn job that will periodically rebuild the tree
        if asynchronously:
            self.spawn(self.build_tree)
        else:
            self.build_tree()

    def key(self):
        return 'abc'

    def name(self):
        return 'ABC groups'

    @classmethod
    def create(cls, options, log):
        assert isinstance(options, dict)
        return cls(
            options['oauth'],
            asynchronously=options.get('async', False),
            log=log.getChild('groups.abc'),
        )

    def _fetch(self, url, params=None):
        return requests.get(
            url=url,
            headers={
                "Accept": "application/json",
                'Authorization': 'OAuth ' + self.oauth,
            },
            params=params or {},
        )

    def fill_services_tree(self, tree):
        url = self.API_SERVICES_URL
        m = datetime.now() - datetime.now()
        while url:
            n = datetime.now()
            req = self._fetch(url)
            d = datetime.now() - n
            m += d
            self.log.info("loaded %s page %s", d, url)

            if not req.ok:
                self.log.debug("page %s failed: %s %s", url, req.status_code, req.reason)
                # TODO retry?
                break

            try:
                data = req.json()
                if not isinstance(data, dict) or not data.get('results'):
                    break

                for result in data['results']:
                    slug = 'svc_' + result['slug']
                    svc_id = str(result['id'])
                    if slug not in tree:
                        tree[svc_id] = tree[slug] = Group(set(), set())

                    for ancestor in result['ancestors']:
                        aslug = 'svc_' + ancestor['slug']
                        asvc_id = str(ancestor['id'])
                        if aslug not in tree:
                            tree[asvc_id] = tree[aslug] = Group(set(), set())
                        tree[aslug].descendants.add(slug)

                url = data['next']
            except Exception:
                # TODO retry?, log
                self.log.exception("failed to fetch page %s", url)
                break
        self.log.info("filled pages %s", m)

    def fill_members_tree(self, tree):
        url = self.API_MEMBERS_URL
        m = datetime.now() - datetime.now()
        while url:
            n = datetime.now()
            req = self._fetch(url)
            d = datetime.now() - n
            m += d
            self.log.info("loaded %s page %s", d, url)

            if not req.ok:
                # TODO retry?
                self.log.debug("page %s failed: %s %s", url, req.status_code, req.reason)
                break

            try:
                data = req.json()
                if not isinstance(data, dict) or not data.get('results'):
                    self.log.warning("url %r invalid: %r", url, data)
                    break

                for result in data['results']:
                    aslug = 'svc_' + result['service']['slug']
                    asvc_id = str(result['service']['id'])
                    if aslug not in tree:
                        tree[asvc_id] = tree[aslug] = Group(set(), set())

                    scope = result['role']['scope']['slug']
                    slug = aslug + '_' + scope

                    if slug not in tree:
                        tree[slug] = Group(set(), set())

                    tree[aslug].descendants.add(slug)
                    tree[slug].logins.add(result['person']['login'])

                url = data['next']
            except Exception:
                # TODO retry?
                self.log.exception("failed to fetch page %s", url)
                break
        self.log.info("filled pages %s", m)

    def build_tree(self):
        tree = {}
        self.fill_services_tree(tree)
        self.log.info("filled services list")
        self.fill_members_tree(tree)
        self.log.info("filled members list")

        self.groups = tree

    def find_users(self, name):
        sg = self.groups
        if name not in sg:
            return

        users = set()
        group = sg[name]
        users |= group.logins

        for descendant in group.descendants:
            users |= sg[descendant].logins

        return users
