import logging
import requests
from collections import defaultdict
from datetime import datetime

from .abstract import Group, GroupSource


class StaffGroups(GroupSource):
    API_GROUPS_URL = "https://staff-api.yandex-team.ru/v3/groups?_pretty=0&is_deleted=false&_sort=id"
    API_URL = "https://staff-api.yandex-team.ru/v3/groupmembership?_pretty=0&_sort=id"

    def __init__(self, oauth, asynchronously=False, log=None):
        self.oauth = oauth
        self.groups = {}
        self.log = log or logging.getLogger('staff')
        # TODO spawn job that will periodically rebuild the tree
        if asynchronously:
            self.spawn(self.build_tree)
        else:
            self.build_tree()

    def key(self):
        return 'staff'

    def name(self):
        return 'Staff groups'

    @classmethod
    def create(cls, options, log):
        assert isinstance(options, dict)
        return cls(
            options['oauth'],
            asynchronously=options.get('async', False),
            log=log.getChild('groups.staff'),
        )

    def _fetch(self, url, params=None):
        return requests.get(
            url=url,
            headers={
                "Accept": "application/json",
                'Authorization': 'OAuth ' + self.oauth,
            },
            params=params or {},
        )

    def build_tree(self):
        tree = defaultdict(lambda: Group(set(), set()))
        self.fill_services_tree(tree)
        self.fill_users_tree(tree)
        self.groups = tree

    def fill_services_tree(self, tree):
        url = self.API_GROUPS_URL
        chunk_size = 20000
        m = datetime.now() - datetime.now()
        args = {
            "_limit": str(chunk_size),
            '_fields': ','.join((
                'id',
                'url',
                'ancestors.is_deleted',
                'ancestors.url',
            )),
        }
        while True:
            n = datetime.now()
            req = self._fetch(url, args)
            d = datetime.now() - n
            m += d
            self.log.info("loaded %s page %s, %s", d, url, args)

            if not req.ok:
                self.log.debug("page %s failed: %s %s", url, req.status_code, req.reason)
                # TODO retry?
                break

            try:
                data = req.json()
                if not isinstance(data, dict) or not data.get('result'):
                    break

                for result in data['result']:
                    u = result['url']
                    for ancestor in result['ancestors']:
                        if ancestor['is_deleted']:
                            continue

                        aurl = ancestor['url']
                        tree[aurl].descendants.add(u)

                args['_query'] = 'id>%s' % (data['result'][-1]['id'],)

            except Exception:
                # TODO retry?
                self.log.exception("failed to fetch page %s", url)
                break
        self.log.info("filled pages %s", m)

    def fill_users_tree(self, tree):
        url = self.API_URL
        chunk_size = 20000
        m = datetime.now() - datetime.now()
        args = {
            'group.is_deleted': 'false',
            'person.is_deleted': 'false',
            'person.official.is_dismissed': 'false',
            '_fields': ','.join((
                'id',
                'person.login',
                'group.url',
            )),
            '_limit': str(chunk_size),
        }
        while True:
            n = datetime.now()
            req = self._fetch(url, args)
            d = datetime.now() - n
            m += d
            self.log.info("loaded %s page %s, %s", d, url, args)

            if not req.ok:
                self.log.debug("page %s failed: %s %s", url, req.status_code, req.reason)
                break

            try:
                data = req.json()

                if not isinstance(data, dict) or not data.get('result'):
                    return

                for result in data['result']:
                    login = result['person']['login']
                    u = result['group']['url']

                    tree[u].logins.add(login)

                args['_query'] = 'id>%s' % (data['result'][-1]['id'],)

            except Exception:
                # TODO retry?, log
                self.log.exception("failed to fetch page %s", url)
                break
        self.log.info("filled pages %s", m)

    def find_users(self, name):
        if name not in self.groups:
            return

        users = set()
        group = self.groups[name]
        if group.logins:
            users |= group.logins

        for descendant in group.descendants:
            subgroup = self.groups[descendant]
            if subgroup.logins:
                users |= subgroup.logins

        return users
