from __future__ import absolute_import

import logging
import smtplib
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

from .abstract import Notifier, NotifyException


class EmailNotifier(Notifier):
    def __init__(self, from_email, default_subject, smtp_server):
        self.from_email = from_email
        self.default_subject = default_subject
        self.smtp_server = smtp_server
        self.log = logging.getLogger("email")

    def __str__(self):
        return "EmailNotifier(from_email=%r, default_subject=%r, smtp_server=%r)" % (
            self.from_email, self.default_subject, self.smtp_server)

    @classmethod
    def create(cls, options):
        assert isinstance(options, dict), "options should be dict but %r" % (type(options),)
        from_email = options.get('from', "Qnotifier <devnull-noreply@yandex-team.ru>")
        default_subject = options.get("default_subject", "Notification from Qloud")
        smtp_server = options.get('smtp', 'localhost')
        return cls(from_email, default_subject, smtp_server)

    def key(self):
        return "email"

    def name(self):
        return "Email"

    def read_user_address(self, data):
        if isinstance(data, dict) and 'email' in data:
            addr = data['email']
            first = data.get('first_name')
            last = data.get('last_name')
            if first and last:
                addr = '%s %s <%s>' % (first, last, addr)
            elif first or last:
                addr = '%s <%s>' % (first or last, addr)
            return addr

    def notify_user(self, address, message):
        options = message['options'] or {}
        html_version = options.get('message-html')
        plain_text = message['message']

        msg = MIMEMultipart("alternative")
        msg['Subject'] = options.get('subject', self.default_subject)
        msg['From'] = options.get('from_email', self.from_email)
        msg['To'] = address
        msg.attach(MIMEText(plain_text, 'plain'))
        if html_version:
            msg.attach(MIMEText(html_version, 'html'))

        for header in options.get('headers', []):
            if not isinstance(header, dict) or not 'name' in header or not 'value' in header:
                continue

            if header['name'] == 'To':
                continue
            msg[header['name']] = header['value']

        self.log.info("send message to(%r) subject(%r) message(%r)" % (msg['To'], msg['Subject'], plain_text))

        try:
            with smtplib.SMTP(self.smtp_server) as smtp:
                smtp.send_message(msg)
        except Exception as e:
            raise NotifyException("Email send failed: %s" % (e,))
