from .abstract import Notifier

import logging

import flask
import telegram


class TelegramNotifier(Notifier):
    MAX_MESSAGE_LENGTH = 4096
    SOFT_SPLIT_SYMBOL = '\u0082'

    def __init__(self, db, api_key, log=None):
        self.db = db
        self.log = log or logging.getLogger('tg')

        self.bot = telegram.Bot(token=api_key)

    def key(self):
        return "telegram"

    def name(self):
        return "Telegram"

    def _find_user(self, address):
        query = "SELECT chat_id FROM tg_chats WHERE nickname = %s"
        db = self.db.getconn('notify')
        try:
            with db, db.cursor() as cur:
                cur.execute(query, (address,))
                row = cur.fetchone()
                return int(row[0]) if row else None
        finally:
            self.db.putconn(db, 'notify')

    @classmethod
    def create(cls, options):
        assert isinstance(options, dict), "options should be dict but %r given" % (type(options),)
        db = flask.current_app.database
        key = options['api_key']
        return cls(db, key)

    def read_user_address(self, data):
        if isinstance(data, dict) and 'nickname' in data:
            return data['nickname']

    def notify_user(self, address, message):
        if address is None:
            return
        chat_id = self._find_user(address)
        self.log.info("notifying %s to %s", address, chat_id)
        if chat_id is None:
            return

        options = message['options'] or {}

        mode = options.get('message-format')
        mformat = None
        if mode == 'html':
            mformat = telegram.ParseMode.HTML
            data = options.get('message-html', message)
        elif mode == 'markdown':
            mformat = telegram.ParseMode.MARKDOWN
            data = options.get('message-markdown', message)
        else:
            data = message['message']

        remainder = None
        chunk = data
        if len(data) > self.MAX_MESSAGE_LENGTH:
            splitpos = data.rfind(self.SOFT_SPLIT_SYMBOL, 0, self.MAX_MESSAGE_LENGTH)
            if splitpos != -1:
                chunk = data[:splitpos]
                remainder = data[splitpos + 1:]  # remove split symbol
            else:
                chunk = data[:self.MAX_MESSAGE_LENGTH]
                remainder = data[self.MAX_MESSAGE_LENGTH:]

        silent = not options.get("urgent", True)

        self.bot.send_message(chat_id=chat_id, text=chunk, parse_mode=mformat, disable_notification=silent)

        if remainder:
            if mode == 'html':
                message.setdefault('options', {})['message-html'] = remainder
            elif mode == 'markdown':
                message.setdefault('options', {})['message-markdown'] = remainder
            else:
                message['message'] = remainder
            return message
