import logging
import threading
from queue import Queue, Empty

import flask

from .abstract import EventQueue
from .. import notifiers


class EmbeddedQueue(EventQueue):
    def __init__(self, notifiers=None, log=None):
        self.log = log or logging.getLogger('emqueue')
        self.stopped = False
        self.queue = Queue()
        self.notifiers = {}
        self._set_notifiers(notifiers)

        self.worker = threading.Thread(target=self.process_queue, args=(flask.current_app._get_current_object(),), daemon=True)
        self.worker.start()

    def _set_notifiers(self, notifiers_list):
        if notifiers_list is None:
            # oh, hack
            notifiers_list = flask.current_app.config['WORKER']['notifiers']

        for item in notifiers_list:
            klass = item['classname']
            params = item['options']
            source_type = getattr(notifiers, klass, None)
            if source_type is None:
                raise TypeError("notifiers config has invalid class: %r" % (klass,))

            try:
                source = source_type.create(params)
            except Exception as e:
                raise ValueError("notifier source %r cannot be created: %s" % (klass, e))
            else:
                self.log.info("initialized: %s", source)
                self.notifiers[source.key()] = source

    @classmethod
    def create(cls, options, log):
        return cls(log=log.getChild('queues.embedded'))

    def key(self):
        return 'embedded'

    def name(self):
        return 'Embedded queue'

    def put(self, data):
        self.queue.put(data)

    def stop(self):
        self.stopped = True

    def process_queue(self, app):
        with app.app_context():
            while not self.stopped:
                try:
                    msg = self.queue.get(5)
                except Empty:
                    continue

                msg['attempt'] += 1

                notifier = self.notifiers.get(msg['type'])
                if notifier is None:
                    self.log.info("unknown notifier type: %r (attempt #%s)", msg['type'], msg['attempt'])
                    self.push_back(msg)
                    return

                address = notifier.read_user_address(msg['address'])
                if address is None:
                    self.log.info("cannot parse address: %r (attempt #%s)", msg['address'], msg['attempt'])
                    self.push_back(msg)
                    return

                options = msg['options'] or {}

                try:
                    notifier.notify_user(address, msg['message'], **options)
                except notifiers.NotifyException:
                    self.log.exception("failed to notify (type %r, attempt #%s)", msg['type'], msg['attempt'])
                    self.push_back(msg)

    def push_back(self, msg):
        if msg['attempt'] > 100:  # FIXME configurable value
            self.log.warning("failed to deliver message (type %r, attempt #%s), dropping",
                             msg['type'], msg['attempt'])
            return

        self.queue.put(msg)
