import time
import logging

import msgpack
from concurrent import futures

from .abstract import EventQueue

from ..logbroker import PQClient


TIMEOUT = 60


class LogbrokerQueue(EventQueue):
    def __init__(self, pq_config, writer_config, initial_seq=None, log=None):
        self.api = PQClient(pq_config, writer_config=writer_config)
        self.api.start().result(timeout=300)
        self.log = log or logging.getLogger('lbqueue')
        self.writer = None
        self.seq = int(initial_seq or time.time())

    @classmethod
    def create(cls, options, log):
        host = options['host']
        port = options['port']
        topic = options['topic']
        client_id = options['client_id']
        service_id = options['service_id']
        source_id = options['source_id']
        secret = options['secret']

        return cls(
            {'host': host, 'port': port, 'topic': topic},
            {'client_id': client_id, 'secret': secret, 'service_id': service_id, 'source_id': source_id},
            log=log.getChild('queues.lb'),
        )

    def key(self):
        return 'logbroker'

    def name(self):
        return 'Logbroker queue'

    def put(self, data):
        try:
            self._put(data)
        except Exception:
            if self.writer is None:
                self.log.debug("writer was invalidated, will reestablish connection and retry")
                self._put(data)
            else:
                raise

    def _put(self, data):
        deadline = time.time() + TIMEOUT
        if self.writer is None or self.writer.stop_future.done():
            self.create_writer()

        data = msgpack.dumps(data, use_bin_type=True)
        seq = self.next_seq()

        timeout = deadline - time.time()
        if timeout < 0.01:
            raise Exception("Failed to write message #%s, timeout reached" % (seq,))

        try:
            future = self.writer.write(seq, data)
        except Exception as e:
            self.log.info("write failed: %s", e)
            self.writer = None
            raise Exception("Failed to put message #%s, writer is dead", seq)

        futures.wait((self.writer.stop_future, future),
                     timeout=timeout, return_when=futures.FIRST_COMPLETED)

        if self.writer.stop_future.done():
            self.log.info("writer is terminated: %s", self.writer.stop_future.result(timeout=0))
            self.writer = None
            raise Exception("Failed to put message #%s, writer is dead", seq)

        elif not future.done():
            future.cancel()  # just in case
            self.log.debug("failed to write message #%s", seq)
            raise Exception("Failed to write message #%s to queue" % (seq,))

        self.log.debug("message #%s written", seq)

    def create_writer(self):
        writer = self.api.create_writer()
        futures.wait((writer.start_future, writer.stop_future),
                     timeout=TIMEOUT, return_when=futures.FIRST_COMPLETED)

        if writer.stop_future.done():
            raise Exception("failed to create writer: %s" % (writer.stop_future.result(timeout=0),))

        if not writer.start_future.done():
            writer.stop()
            raise Exception("failed to initializer writer: %s" % (writer.stop_future.result(timeout=0),))

        self.writer = writer

    def next_seq(self):
        seq, self.seq = self.seq, self.seq + 1
        return seq

    def stop(self):
        self.api.stop()
