import os
import sys
import time
import struct
import socket
import logging
from functools import partial


# FIXME (torkve) when ported to arcadia, use standard api.logger

daemonName = sys.argv[0]


def get_root():
    path = os.getenv('LOGGER_SOCKET_PATH', None)
    if path is not None:
        return path

    from api.srvmngr import getRoot
    return os.path.join(getRoot(), 'var', 'log')


class Logger(object):
    struct = struct.Struct('!IIIiI')
    instance = None

    @classmethod
    def make(cls, udp_socket_path=None):
        instance = cls.instance

        if udp_socket_path is None and instance is None:
            raise RuntimeError("Logger is not connected yet")
        elif udp_socket_path is None:
            return instance
        elif instance and instance.dest == udp_socket_path and not instance.closed:
            return instance

        if instance and not instance.closed:
            instance.close()

        instance = cls.instance = cls(udp_socket_path)
        return instance

    def __init__(self, udp_socket_path):
        self.sock = socket.socket(socket.AF_UNIX, socket.SOCK_DGRAM)
        self.sock.settimeout(1)
        self.closed = False
        self.dest = udp_socket_path

    def close(self):
        self.closed = True
        self.sock.close()

    def log(self, app, namespace, level, message, filename=None):
        max_packet = 2 ** 16 - 28
        if filename is None:
            filename = ''

        if sys.version_info[0] == 2:
            if isinstance(message, unicode):
                message = message.encode('utf-8')

        if sys.version_info[0] == 3:
            if isinstance(app, str):
                app = app.encode('utf-8')
            if isinstance(namespace, str):
                namespace = namespace.encode('utf-8')
            if isinstance(filename, str):
                filename = filename.encode('utf-8')
            if isinstance(message, str):
                message = message.encode('utf-8')

        desclen = self.struct.size + len(app) + len(namespace) + len(filename)
        if len(message) + desclen > max_packet:
            message = message[:max_packet - desclen]

        header = self.struct.pack(len(app), len(namespace), len(filename), level, len(message))

        message = b''.join((header, app, namespace, filename, message))

        self.sock.sendto(message, self.dest)


def _format(app, level, source, message):
    ct = time.time()
    if sys.version_info[0] == 3:
        msecs = (ct - int(ct)) * 1000
    else:
        msecs = (ct - long(ct)) * 1000
    heading = '%s.%03d %2s [%s%s]  ' % (
        time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(ct)),
        msecs, level, app, source
    )

    result = []
    for idx, line in enumerate(message.strip().split('\n')):
        if idx == 0:
            result.append(' '.join((heading, line)))
        else:
            result.append(' '.join((' ' * len(heading), line.rstrip())))

    return '\n'.join(result)


def raw(level, message, source=None, app='unknown', filename=None):
    """
    raw logging function - use it if you need non existent error level.
    sends :arg str message: with :arg int level: and :arg str app: to
    logger service, that will write it into :arg str filename:.
    optionally duplicates message to stdout if SKYNET_LOG_STDOUT is set.
    if it failes to send message - tries to write to 'fallback_logfile'.
    """
    source = source or daemonName
    src = ':%s' % source if source != 'root' else ''
    try:
        if os.getenv('SKYNET_LOG_STDOUT', False) == '1':
            print(_format(app, level, src, message))
        Logger.make().log(app, source, level, message, filename)
    except Exception as ex:
        # fallback mode if we failed to log exception
        # bad: file path is copy-pasted from server implementation
        # in 99% of cases error was "no buffer space available", which
        # means udp sending failed.
        if filename is None:
            filename = 'logfile'

        if isinstance(level, int):
            if level == 0:
                level = '[----]'
            elif level == 9:
                level = '[PROT]'
            elif level == 10:
                level = '[DEBG]'
            elif level == 20:
                level = '[INFO]'
            elif level == 30:
                level = '[WARN]'
            elif level == 40:
                level = '[ERRR]'
            elif level == 50:
                level = '[CRIT]'

        try:
            root = get_root()
        except Exception:
            # we know nothing about skynet so cannot log directly
            return

        try:
            try:
                formatted = _format(app, level, src, message)
            except:
                formatted = '{0}:{1}:{2}'.format(app, level, message)

            fd = os.open(os.path.join(root, filename), os.O_APPEND | os.O_WRONLY)
            try:
                logfile = os.fdopen(fd, 'a')
            except:
                os.close(fd)

            try:
                logfile.write(formatted + '  [directly logged, because of "' + str(ex) + '"] \n')
            finally:
                logfile.close()
        except EnvironmentError:
            pass


class SkynetLoggingHandler(logging.Handler):
    appName = None

    def __init__(self, app='unknown', filename=None, udp_socket_path=None, *args, **kwargs):
        self.appName = app
        self.filename = filename
        if udp_socket_path is not None:
            Logger.make(udp_socket_path)
        logging.Handler.__init__(self, *args, **kwargs)

    def emit(self, record):
        raw(record.levelno, self.format(record), record.name, self.appName, filename=self.filename)

    def close(self):
        logging.Handler.close(self)
        if Logger.instance:
            Logger.instance.close()
