import logging
from collections import defaultdict

import flask

import telegram
from telegram.ext import Updater, CommandHandler


class TelegramBot:
    def __init__(self, options, unistats=None, log=None):
        self.log = log or logging.getLogger('tg')
        self.unistats = unistats if unistats is not None else defaultdict(int)
        self.db = flask.current_app.database
        self.bot = telegram.Bot(token=options['api_key'])
        self.updater = Updater(bot=self.bot)
        start_handler = CommandHandler('start', self._register_user)
        unregister_handler = CommandHandler('unregister', self._unregister_user)
        self.updater.dispatcher.add_handler(start_handler)
        self.updater.dispatcher.add_handler(unregister_handler)

    def start(self):
        self.updater.start_polling()

    def shutdown(self):
        self.updater.stop()

    def _register_user(self, bot, update):
        user = update.message.from_user
        chat_id = update.message.chat_id
        if user is None or user.username is None:
            bot.send_message(chat_id=chat_id, text="Your login not found, soryan")
            return

        query = "INSERT INTO tg_chats (nickname, chat_id) VALUES (%s, %s) ON CONFLICT (nickname) DO UPDATE set chat_id = excluded.chat_id"
        db = self.db.getconn()
        try:
            with db, db.cursor() as cur:
                try:
                    cur.execute(query, (user.username, str(chat_id)))
                except Exception as e:
                    bot.send_message(
                        chat_id=chat_id,
                        text=f"Registration failed.\nError: {e}"
                    )
                else:
                    bot.send_message(
                        chat_id=chat_id,
                        text="You are registered.\nI'll contact you in case of need.",
                    )
        finally:
            self.db.putconn(db)

    def _unregister_user(self, bot, update):
        user = update.message.from_user
        chat_id = update.message.chat_id
        if user is None or user.username is None:
            bot.send_message(chat_id=chat_id, text="Your login not found, soryan")
            return

        query = "DELETE FROM tg_chats WHERE nickname = %s OR chat_id = %s"
        db = self.db.getconn('bot')
        try:
            with db, db.cursor() as cur:
                try:
                    cur.execute(query, (user.username, str(chat_id)))
                except Exception as e:
                    bot.send_message(
                        chat_id=chat_id,
                        text=f"Unregistration failed.\nError: {e}"
                    )
                else:
                    bot.send_message(
                        chat_id=chat_id,
                        text="You are unregistered.\nNo further notifications will be sent via telegram.",
                    )
        finally:
            self.db.putconn(db)

