import requests

from .abstract import User, UserSource


class StaffUsers(UserSource):
    API_URL = "https://staff-api.yandex-team.ru/v3/persons?_pretty=0&_one=1"

    def __init__(self, oauth, log):
        self.log = log
        self.oauth = oauth

    def key(self):
        return 'staff'

    def name(self):
        return 'Staff users'

    @classmethod
    def create(cls, options, log):
        assert isinstance(options, dict)

        return cls(options['oauth'], log.getChild('users.staff'))

    def find_user(self, login):
        url = self.API_URL
        params = {
            'login': login,
            'is_deleted': 'false',
            'official.is_dismissed': 'false',
            '_fields': 'accounts,groups,phones,work_email,name',
        }
        req = requests.get(
            url=url,
            headers={
                "Accept": "application/json",
                'Authorization': 'OAuth ' + self.oauth,
            },
            params=params,
        )

        if not req.ok:
            self.log.warning("Request %r params=%r failed: %r", url, params, req.raw)
            return

        try:
            user = req.json()

            if not isinstance(user, dict):
                return

            first_name = user['name']['first']['en']
            last_name = user['name']['last']['en']

            # TODO support multiple addresses of each type
            telegram = list(filter(lambda acc: acc['type'] == 'telegram' and not acc['private'], user['accounts']))
            telegram = telegram[0]['value'] if telegram else None
            email = user['work_email']
            sms_phones = filter(
                lambda phone: phone['protocol'] in ('all', 'sms') and phone['type'] == 'mobile',
                user['phones']
            )
            sms_phones = sorted(sms_phones,
                                key=lambda phone: (phone['kind'] != 'monitoring', not phone['is_main']))
            sms_phone = sms_phones[0]['number'] if sms_phones else None

            voice_phones = filter(
                lambda phone: phone['protocol'] in ('all', 'voice'),
                user['phones']
            )
            voice_phones = sorted(voice_phones,
                                  key=lambda phone: (phone['kind'] != 'monitoring', not phone['is_main']))
            voice_phone = voice_phones[0]['number'] if voice_phones else None

            groups = [group['group']['url'] for group in user['groups']]

            result = User(login)
            result.set_address_for('email', {'email': email, 'first_name': first_name, 'last_name': last_name})
            result.set_address_for('telegram', {'nickname': telegram, 'login': login})
            result.set_address_for('sms', sms_phone)
            result.set_address_for('voice', voice_phone)
            result.set_address_for('groups', groups)  # FIXME

            return result
        except Exception:
            self.log.exception("User %r parse failed", login)
            return

    def check_telegram_authentication(self, login):
        raise NotImplementedError()
