from urllib.parse import urlencode
from functools import partial
from collections import OrderedDict

import logging
import requests
import flask

from ..tvm import make_tvm, BlackboxClientId


__all__ = ['Yauth', 'AuthenticationError', 'yauth_required']

X_YA_SERVICE_TICKET_HEADER = "X-Ya-Service-Ticket"


class YaAuthError(RuntimeError):
    pass


class AuthenticationError(YaAuthError):
    pass


class NotAuthorizedError(YaAuthError):
    pass


class YaPermissionError(YaAuthError):
    pass


class Yauth(object):
    def __init__(self, app=None):
        if app:
            self.init_app(app)

    def init_app(self, app):
        if 'yauth' in app.extensions or not app.config['AUTHORIZER']:
            return

        # TODO reapply allowed_clients without restart

        log = app.logger.getChild('yauth')

        my_client_id = app.config['AUTHORIZER']['client_id']
        log.debug("my client_id: %r", my_client_id)

        secret = app.config['AUTHORIZER']['secret']

        allowed_clients = app.config['AUTHORIZER']['allowed_clients']
        log.debug("allowed_clients: %r", allowed_clients)

        app.yauth_tvm = make_tvm(my_client_id, secret, prod=False, label="Yauth")
        app.yauth_tvm.allowed_clients.update(allowed_clients)

        session = requests.Session()
        adapter = requests.adapters.HTTPAdapter(
            max_retries=requests.packages.urllib3.Retry(
                **app.config['BLACKBOX_RETRY_PARAMS']
            )
        )
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        app.blackbox_requests_session = session

        app.extensions['yauth'] = self
        app.before_request(partial(self.authorize, app))

    def get_auth_requirements(self, app):
        req = flask.request
        view_func = app.view_functions.get(req.endpoint)
        if hasattr(view_func, 'yauth_user'):
            return view_func.yauth_user, view_func.yauth_service

        view_class = getattr(view_func, 'view_class', None)
        if view_class is None:
            return None, None
        if hasattr(view_class, 'yauth_user'):
            return view_class.yauth_user, view_class.yauth_service

        method = getattr(view_class, req.method.lower(), None)
        return getattr(method, 'yauth_user', None), getattr(method, 'yauth_service', None)

    def authorize(self, app):
        log = app.logger.getChild('yauth')

        log.debug("=> authorize")
        flask.g.yauth_service_id = None
        flask.g.yauth_username = None
        flask.g.yauth_usergroups = []

        user_required, service_required = self.get_auth_requirements(app)
        log.debug("user_required=%r, service_required=%r", user_required, service_required)

        if service_required and X_YA_SERVICE_TICKET_HEADER in flask.request.headers:
            return check_tvm(app)

        # TODO: check if it is safe to allow Cookie/OAuth access here
        if user_required and 'Authorization' in flask.request.headers:
            log.debug(": user_required and 'Authorization' in flask.request.headers")
            return check_oauth(app)
        if user_required and 'Session_id' in flask.request.cookies:
            log.debug(": user_required and 'Session_id' in flask.request.cookies")
            return check_cookie(app)

        if user_required or service_required:
            raise NotAuthorizedError("No auth provided")


def check_tvm(app):
    service_ticket = flask.request.headers[X_YA_SERVICE_TICKET_HEADER]
    ticket = app.yauth_tvm.parse_service_ticket(service_ticket)
    if ticket is None:
        raise AuthenticationError("TVM ticket is invalid")
    flask.g.yauth_service_id = ticket.src


def get_oauth_from_ssh(auth_header, app):
    header, login, ts, signature = auth_header.split(' ', 3)

    data = {
        'grant_type': 'ssh_key',
        'client_id': app.config['AUTHORIZER']['oauth_client_id'],
        'client_secret': app.config['AUTHORIZER']['oauth_client_secret'],
        'login': login,
        'ts': int(ts),
        'ssh_sign': signature,
    }
    url = app.config['OAUTH_TOKEN_URL']

    result = requests.post(url, data=data, timeout=app.config['BLACKBOX_TIMEOUT'])
    if result.ok:
        return f'Bearer {result.json()["access_token"]}'
    else:
        raise NotAuthorizedError(result.text)

def check_oauth(app):
    auth_header = flask.request.headers['Authorization']
    if auth_header.startswith('SSH '):
        try:
            token = get_oauth_from_ssh(auth_header, app)
        except Exception as exc:
            raise NotAuthorizedError('SSH to OAuth exchange failed: %s' % (exc,)).with_traceback(exc.__traceback__)
    else:
        token = auth_header

    # OAuth authorization
    # https://doc.yandex-team.ru/blackbox/reference/method-oauth.xml
    args = OrderedDict((
        ('method', 'oauth'),
        ('userip', flask.request.access_route[-1]),
        ('format', 'json'),
    ))
    bb_id = str(BlackboxClientId.ProdYateam.value)
    headers = {
        'Authorization': token,
        'Accept': 'application/json',
        X_YA_SERVICE_TICKET_HEADER: app.yauth_tvm.get_service_tickets(bb_id)[bb_id],
    }
    return get_blackbox(app, args, headers)


def check_cookie(app):
    session_id = flask.request.cookies.get('Session_id', None)
    session_id2 = flask.request.cookies.get('sessionid2', None)

    # cookie-based authorization
    host = flask.request.environ.get('HTTP_HOST')
    if not host:
        host = app.config['SERVER_NAME']
    host = host.split(':')[0]
    # http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml
    args = OrderedDict((
        ('method', 'sessionid'),
        ('sessionid', session_id),
        ('userip', flask.request.access_route[-1]),
        ('host', host),
        ('format', 'json'),
    ))
    if session_id2:
        args['sslsessionid'] = session_id2
    bb_id = str(BlackboxClientId.ProdYateam.value)
    headers = {
        'Accept': 'application/json',
        X_YA_SERVICE_TICKET_HEADER: app.yauth_tvm.get_service_tickets(bb_id)[bb_id],
    }
    return get_blackbox(app, args, headers)


def get_blackbox(app, args, headers):
    log = app.logger.getChild('yauth')

    blackbox_uri = '%s?%s' % (app.config['BLACKBOX_URI'], urlencode(args))
    try:
        result = app.blackbox_requests_session.get(
            blackbox_uri,
            timeout=app.config['BLACKBOX_TIMEOUT'],
            verify=True,
            headers=headers,
        ).json()
    except Exception as exc:
        raise AuthenticationError("Blackbox authentication failed: %s" % (exc,)).with_traceback(exc.__traceback__)

    if result['error'] == 'OK':
        if result['status']['value'] == 'VALID':
            flask.g.yauth_username = result['login']
            try:
                flask.g.yauth_usergroups = get_groups(flask.g.yauth_username, app.config)
            except Exception:
                pass
                # flask.g.group_auth_error = {
                #     'exc_class': type(exc).__name__,
                #     'exc_message': str(exc),
                #     'traceback': traceback.format_exc(),
                # }
            log.info("authorized as %r with groups %r",
                     flask.g.yauth_username,
                     flask.g.yauth_usergroups)
            return
        elif result['status']['value'] == 'NEED_RESET':
            url = app.config['RENEW_AUTH_COOKIE_URI']
            args = urlencode({'retpath': flask.request.url})
            return flask.redirect('%s?%s' % (url, args))

    if result.get('exception'):
        log.error("blackbox failure: %s", result)

    raise AuthenticationError("authentication failure: %s" % (result,))


# FIXME we already have one module to fetch user groups
def get_groups(username, cfg):
    # https://staff-api.yandex-team.ru/v3/persons?_doc=1
    params = {'login': username,
              '_one': '1',
              'official.is_dismissed': 'false',
              '_fields': ','.join(('groups.group.url',
                                   'department_group.url',
                                   'department_group.ancestors.url'))}
    resp = requests.get(
        cfg['STAFF_URI'] + '/persons',
        params=params,
        headers={
            'Authorization': 'OAuth %s' % (cfg['OAUTH_YANDEX_TOKEN']),
            'Accept': 'application/json',
        },
        timeout=cfg['STAFF_TIMEOUT'],
        allow_redirects=False
    )
    resp.raise_for_status()
    resp = resp.json()
    groups = set(rec['group']['url'] for rec in resp['groups'])
    groups.add(resp['department_group']['url'])
    groups.update(rec['url']
                  for rec in resp['department_group'].get('ancestors', []))
    return sorted(groups)


def resolve_service_administration(name, cfg):
    # resolve servicerole|service -> 'service'_administration
    params = {
        '_one': '1',
        'is_deleted': 'false',
        '_fields': ','.join((
            'type',
            'role_scope',
            'parent',
            'url',
        ))
    }
    try:
        int(name)
    except ValueError:
        params['url'] = name
    else:
        params['service.id'] = name

    resp = requests.get(
        cfg['STAFF_URI'] + '/groups',
        params=params,
        headers={
            'Authorization': 'OAuth %s' % (cfg['OAUTH_YANDEX_TOKEN']),
            'Accept': 'application/json',
        },
        timeout=cfg['STAFF_TIMEOUT'],
        allow_redirects=False
    )
    resp.raise_for_status()
    resp = resp.json()
    if resp['type'] == 'service':
        return resp['url'] + '_administration'
    elif resp['type'] == 'servicerole':
        if resp['role_scope'] == 'administration':
            return resp['url']
        parent = resp['parent']
        if parent:
            return parent['url'] + '_administration'

    return YaPermissionError("Cannot resolve modify permissions for group %r" % (name,))


def yauth_required(fn=None, user=False, service=False):
    assert user or service, "At least one authorization method must be specified"

    if fn is None:
        return partial(yauth_required, user=user, service=service)

    fn.yauth_user = user
    fn.yauth_service = service
    return fn


def check_permissions_for(name, is_group):
    app = flask.current_app
    g = flask.g
    if (
        # if auth disabled
        'yauth' not in app.extensions
        # services can change everything
        or g.yauth_service_id
        # user can change their own data
        or (not is_group and name == g.yauth_username)
    ):
        return

    if is_group:
        required_role = resolve_service_administration(name, app.config)
        if required_role in g.yauth_usergroups:
            return

    # TODO allow one user to edit subscriptions of another user?

    message = "User %r is not allowed to change %s %r" % (
        g.yauth_username,
        "group" if is_group else "user",
        name,
    )
    app.logger.getChild('yauth').debug("%s", message)

    raise YaPermissionError(message)
