import sys
import time
import logging
import traceback
from functools import partial, partialmethod

from .console import is_terminal


# Patch logging to have short level names
_levelNameAliases = {
    # LEVEL:    (1char, 3 chars, 4 chars, full)
    'DEBUG':    ('D',   'DBG',   'DEBG',  'DEBUG'),
    'INFO':     ('I',   'INF',   'INFO',  'NORMAL'),
    'WARNING':  ('W',   'WRN',   'WARN',  'WARNING'),
    'ERROR':    ('E',   'ERR',   'ERRR',  'ERROR'),
    'CRITICAL': ('C',   'CRI',   'CRIT',  'CRITICAL'),
    'NOTSET':   ('-',   '---',   '----',  'NOTSET'),
}


def rename_levels(chars=None):
    assert chars in (None, 1, 3, 4), 'We only support unlimited and 1-3-4 char themes'

    if chars == 1:
        idx = 0
    elif chars == 3:
        idx = 1
    elif chars == 4:
        idx = 2
    else:
        idx = 3

    for level, aliases in list(_levelNameAliases.items()):
        selectedAlias = aliases[idx]

        levelno = logging.getLevelName(level)
        logging.addLevelName(levelno, selectedAlias)


logging._levelNameAliases = _levelNameAliases
logging.rename_levels = logging.renameLevels = renameLevels = rename_levels


class ExtendedFormatter(logging.Formatter):
    def __init__(
        self,
        fmt='%(asctime)s - %(levelname)s - %(name)s - %(message)s',
        fmtName='%(name)s',
        fmtLevelno='%(levelno)d',
        fmtLevelname='%(levelname)s',
        fmtPathname='%(pathname)s',
        fmtFilename='%(filename)s',
        fmtModule='%(module)s',
        fmtLineno='%(lineno)d',
        fmtFuncName='%(funcName)s',
        fmtCreated='%(created)f',
        fmtAsctime='%(datetime)s,%(msecs)03d',
        fmtDatetime='%Y-%m-%d %H:%M:%S',
        fmtMsecs='%(msecs)d',
        fmtRelativeCreated='%(relativeCreated)d',
        fmtThread='%(thread)d',
        fmtThreadName='%(threadName)s',
        fmtProcess='%(process)d',
        fmtProcessName='%(processName)s',
        fmtHost='%(host)s',
        fmtMessage='%(message)s',
    ):
        self.__dict__.update(locals())

    def formatTime(self, record, datefmt=None):
        ct = self.converter(record.created)
        datetime = time.strftime(datefmt if datefmt is not None else self.fmtDatetime, ct)
        return self.fmtAsctime % {
            'datetime': datetime,
            'msecs': record.msecs
        }

    def formatSimple(self, record, type_):
        default = '%%(%s)s' % (type_, )
        return getattr(self, 'fmt%s' % (type_[0].upper() + type_[1:], ), default) % {
            type_: getattr(record, type_)
        }

    formatName = partialmethod(formatSimple, type_='name')
    formatLevelno = partialmethod(formatSimple, type_='levelno')
    formatLevelname = partialmethod(formatSimple, type_='levelname')
    formatPathname = partialmethod(formatSimple, type_='pathname')
    formatFilename = partialmethod(formatSimple, type_='filename')
    formatModule = partialmethod(formatSimple, type_='module')
    formatLineno = partialmethod(formatSimple, type_='lineno')
    formatFuncName = partialmethod(formatSimple, type_='funcName')
    formatCreated = partialmethod(formatSimple, type_='created')
    formatMsecs = partialmethod(formatSimple, type_='msecs')
    formatRelativeCreated = partialmethod(formatSimple, type_='relativeCreated')
    formatThread = partialmethod(formatSimple, type_='thread')
    formatThreadName = partialmethod(formatSimple, type_='threadName')
    formatProcess = partialmethod(formatSimple, type_='process')
    formatProcessName = partialmethod(formatSimple, type_='processName')
    formatHost = partialmethod(formatSimple, type_='host')
    formatMessage = partialmethod(formatSimple, type_='message')

    def prepare(self, record):
        fmtData = {}
        record.message = record.getMessage()
        if '%(asctime)' in self.fmt:
            fmtData['asctime'] = record.asctime = self.formatTime(record)
        if not hasattr(record, 'host'):  # Beware: hack for local and remote log records mix
            record.host = 'localhost'

        if record.exc_text:
            if not isinstance(record.message, str):
                record.message = str(record.message)
            if not record.message.endswith('\n'):
                record.message += '\n'
            try:
                record.message += record.exc_text
            except UnicodeError:
                # Sometimes filenames have non-ASCII chars, which can lead
                # to errors when s is Unicode and record.exc_text is str
                # See issue 8924
                record.message += record.exc_text.decode(sys.getfilesystemencoding())

        for simpleItem in list(record.__dict__.keys()):
            if simpleItem in fmtData:
                continue
            if '%%(%s)' % simpleItem in self.fmt:
                try:
                    formatter = getattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]))
                except AttributeError:
                    formatter = partial(self.formatSimple, type_=simpleItem)
                    setattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]), formatter)
                fmtData[simpleItem] = formatter(record)

        return fmtData

    def format(self, record):
        fmtData = self.prepare(record)
        return self.fmt % fmtData

    def formatException(self, excInfo):
        return ''.join(traceback.format_exception(*excInfo))


class ColoredFormatter(ExtendedFormatter):
    COLORS = dict(zip(
        ('BLACK', 'RED', 'GREEN', 'YELLOW', 'BLUE', 'MAGENTA', 'CYAN', 'WHITE'),
        [{'num': x} for x in range(30, 38)]
    ))

    ALIASES = {
        'NORMAL': 'INFO',
        'UNKNOWN': 'NOTSET'
    }

    for key, values in _levelNameAliases.items():
        for value in values:
            ALIASES[value] = key

    for color, info in list(COLORS.items()):
        COLORS[color]['normal'] = '\033[0;%dm' % info['num']
        COLORS[color]['bold'] = '\033[1;%dm' % info['num']

    RESET_SEQ = '\033[m'

    def __init__(self, **kwargs):
        # Defaults for colorMap and Hungry levels
        colorMap = {
            'levels': {
                # LEVEL:    (COLOR,    BOLD_OR_NOT)
                'PROTOCOL': ('WHITE',  True),
                'DEBUG':    ('BLACK',  True),
                'INFO':     ('GREEN',  False),
                'WARNING':  ('YELLOW', True),
                'ERROR':    ('RED',    False),
                'CRITICAL': ('RED',    True),
            },
            'parts': {
                # PARTNAME: (COLOR,    BOLD_OR_NOT
                'asctime':  ('BLACK',  True),
                'name':     ('CYAN',   True),
                'host':     ('BLACK',  True),
            },
        }

        hungryLevels = {
            'CRITICAL': ('full', ),             # Critical logs will have colorized full log message by default
            'ERROR': ('message', 'levelname'),  # Error logs will be colorized too (only message part)
        }

        self.useColors = kwargs.pop('useColors', True) and is_terminal()

        for key, value in list(kwargs.items()):
            dict_ = None
            if key.startswith('color'):
                dict_ = colorMap['parts']
                key2 = key[len('color'):].lower()
            elif key.startswith('level'):
                dict_ = colorMap['levels']
                key2 = key[len('level'):].upper()

            if dict_ is None:
                continue

            bold = value.endswith('+')
            value = value.rstrip('+')
            dict_[key2] = (value.upper(), bold)
            kwargs.pop(key)

        for levelName, childNames in kwargs.pop('hungryLevels', {}).items():
            hungryLevels[levelName.upper()] = childNames

        self.colorMap = colorMap
        self.hungryLevels = hungryLevels

        ExtendedFormatter.__init__(self, **kwargs)

    def colorize(self, string, info):
        """ Colorize string.

        :arg info: is a tuple with [color_name, bold_or_not] items.
        """

        return ''.join((
            self.COLORS[info[0]]['normal' if not info[1] else 'bold'],
            string,
            self.RESET_SEQ,
        ))

    def format(self, record):
        if record.exc_info:
            # Cache the traceback text to avoid converting it multiple times
            # (it's constant anyway)
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)

        if not self.useColors:
            return ExtendedFormatter.format(self, record)

        data = self.prepare(record)

        levelname = self.ALIASES.get(record.levelname, record.levelname)
        levelColor = self.colorMap['levels'].setdefault(levelname, None)
        if levelColor is not None:
            levelHungry = self.hungryLevels.setdefault(levelname, ('levelname', ))
            if 'full' in levelHungry:
                # Colorize full message and return immidiately if we are
                # very very hungry level
                return self.colorize(self.fmt % data, levelColor)
        else:
            levelHungry = ()

        # Colorize all specified parts
        for partName, partColor in list(self.colorMap['parts'].items()):
            if partName in levelHungry:
                partColor = levelColor
            if partName not in data:
                continue
            data[partName] = self.colorize(data[partName], partColor)

        # Colorize all hungry parts by level color
        if levelColor is not None:
            for partName in ['levelname'] + list(levelHungry):
                data[partName] = self.colorize(data[partName], levelColor)

        return self.fmt % data
