import os
import sys
from urllib.parse import urljoin
from argparse import ArgumentParser
import textwrap
import warnings

import requests

import ticket_parser2

from ticket_parser2.api.v1 import BlackboxClientId
from tvm2 import TVM2


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('--oauth', action='store_true', default=bool(os.getenv('QNOTIFIER_OAUTH')), help='Test methods, accepting oauth')
    parser.add_argument('--no-oauth', action='store_false', dest='oauth', help="Don't test methods, accepting oauth")
    parser.add_argument('--tvm', action='store_true', default=bool(os.getenv('QNOTIFIER_TVM_SECRET')), help='Test methods, accepting tvm')
    parser.add_argument('--tvm-id', type=int, default=2002140, help='TVM application ID')
    parser.add_argument('--no-tvm', action='store_false', dest='tvm', help="Don't test methods, accepting tvm")
    parser.add_argument('--user', default=os.getlogin(), help='Username to use')
    parser.add_argument('endpoint', help='Address of the API endpoint to test', default='https://qnotifier-test.yandex-team.ru')
    return parser.parse_args()


class ApiSession(requests.Session):
    def __init__(self, endpoint, *args, **kwargs):
        headers = kwargs.pop('headers', {})

        super().__init__(*args, **kwargs)

        for header, value in headers.items():
            self.headers[header] = value

        self.verify = False

        if not endpoint.endswith('/'):
            endpoint += '/'
        self.endpoint = endpoint

    def request(self, method, url, *args, **kwargs):
        url = urljoin(self.endpoint, url)
        print(method, url)
        return super().request(method, url, *args, **kwargs)


def with_tvm(f):
    f.has_tvm = True
    return f


def with_oauth(f):
    f.has_oauth = True
    return f


def tested(f):
    f.tested = True
    f.has_tvm = False
    f.has_oauth = False
    return f


@with_tvm
@with_oauth
@tested
def test_settings(session, args):
    """Test settings update"""

    url = "settings/" + args.user

    payload = {
        "telegram": {
            "enabled": "true",
        },
        "email": {
            "enabled": "true",
        },
    }
    result = session.post(url, json=payload)
    result.raise_for_status()

    result = session.get(url)
    result.raise_for_status()
    response = result.json()
    assert response == payload, "Got " + result.text

    payload = {
        "telegram": {
            "enabled": "false",
        }
    }
    result = session.patch(url, json=payload)
    result.raise_for_status()

    result = session.get(url)
    result.raise_for_status()
    expected = {
        "telegram": {
            "enabled": "false",
        },
        "email": {
            "enabled": "true",
        },
    }

    response = result.json()
    assert response == expected, "Got " + result.text


@with_tvm
@with_oauth
@tested
def test_subscriptions(session, args):
    """Test subscriptions"""

    url = "subscriptions/" + args.user

    payload = {
        "options": {
            "email": {
                "enabled": "true",
            },
            "telegram": {
                "enabled": "true",
            },
        },
        "tags": ["apitest:tag1", "apitest:tag2", "!apitest:tag3"],
    }
    result = session.post(url, json=payload)
    result.raise_for_status()

    payload = {
        "tags": ["apitest:tag1", "apitest:tag2", "!apitest:tag3"],
    }
    result = session.delete(url, json=payload)
    result.raise_for_status()

    payload = {
        "options": {
            "email": {
                "enabled": "true",
            },
            "telegram": {
                "enabled": "true",
            },
        },
        "tags": ["apitest:tag1", "apitest:tag2", "!apitest:tag3"],
    }
    result = session.post(url, json=payload)
    result.raise_for_status()

    payload = {
        "options": {
            "telegram": {
                "enabled": "false",
            },
        },
        "tags": ["apitest:tag1", "apitest:tag2", "!apitest:tag3"],
    }
    result = session.patch(url, json=payload)
    result.raise_for_status()

    expected = [{
        "options": {
            "email": {
                "enabled": "true",
            },
            "telegram": {
                "enabled": "false",
            },
        },
        "tags": ["apitest:tag1", "apitest:tag2", "!apitest:tag3"],
    }]
    result = session.get(url, params={"tags": ["apitest:tag2", "!apitest:tag3", "apitest:tag1"]})
    result.raise_for_status()
    response = result.json()
    assert response['subscriptions'] == expected, "Got " + result.text


@with_tvm
@tested
def test_events(session, args):
    """post an event"""

    message = textwrap.dedent(
        """\
        (This is automatic message, do not reply)

        You QYP vm was evicted successfully. It should now be accessible!

        Evacuation state overview:
        [x] VM stopped and prepared to evacuation
        [x] VM evicted from node it lived on
        [x] VM evacuated to new node and accessible now

        VM props:  vm       : mcsl2
        dc       : test_sas
        node       : iva1-1323.search.yandex.net
        fqdn (r) : iva1-1323-18.mcsl2.sas-test.yp-test.yandex.net
        fqdn (t) : mcsl2.sas-test.yp-test.yandex.net

        Have a nice day!
        """
    )
    markdown_message = textwrap.dedent(
        """\
        *(This is automatic test message, do not reply)*

        You QYP vm was evicted successfully. It should now be accessible!

        Evacuation state overview:
        *[x]* VM stopped and prepared to evacuation
        *[x]* VM evicted from node it lived on
        *[x]* VM evacuated to new node and accessible now

        VM props: vm  : *mcsl2*
        dc : *test-sas*
        node : *iva1-1323.search.yandex.net*
        fqdn (r) : *iva1-1323-18.mcsl2.sas-test.yp-test.yandex.net*
        fqdn (t) : *mcsl2.sas-test.yp-test.yandex.net*

        Have a nice day!
        """
    )
    html_message = textwrap.dedent(
        """\
        <p><b>(This is automatic test message, do not reply)</b></p>

        <p>You QYP vm was evicted successfully. It should now be accessible!</p>

        <p>Evacuation state overview:<br/>
        <b>[x]</b> VM stopped and prepared to evacuation<br/>
        <b>[x]</b> VM evicted from node it lived on<br/>
        <b>[x]</b> VM evacuated to new node and accessible now</p>

        <pre>VM props:  vm       : <b>mcsl2</b>
        dc       : <b>test_sas</b>
        node       : <b>iva1-1323.search.yandex.net</b>
        fqdn (r) : <b>iva1-1323-18.mcsl2.sas-test.yp-test.yandex.net</b>
        fqdn (t) : <b>mcsl2.sas-test.yp-test.yandex.net</b></pre>

        <p>Have a nice day!</p>
        """
    )

    url = "events"
    payload = {
        "tags": ["apitest:tag1", "apitest:tag2"],
        "message": message,
        "forced-users": [],
        "extra": {
            "from_email": "Test Api <devnull@yandex.ru>",
            "subject": "Test Api subject",
            "message-format": "markdown",
            "message-markdown": markdown_message,
            "message-html": html_message,
        },
    }

    result = session.post(url, json=payload)
    result.raise_for_status()


def main():
    args = parse_args()

    if not args.oauth and not args.tvm:
        print("No tokens available")
        return

    if args.oauth:
        headers = {
            'Authorization': os.getenv('QNOTIFIER_OAUTH', ''),
        }
        with ApiSession(endpoint=args.endpoint, headers=headers) as session:
            for function in filter(lambda f: f.has_oauth, TESTED):
                print("=== Testing", function.__name__, "with oauth:", function.__doc__)
                function(
                    session=session,
                    args=args,
                )

    if args.tvm:
        tvm = TVM2(client_id=args.tvm_id, secret=os.getenv('QNOTIFIER_TVM_SECRET', ''), blackbox_client=BlackboxClientId.Prod)
        tvm_ticket = tvm.get_service_tickets('2002376')['2002376']
        headers = {
            'X-Ya-Service-Ticket': tvm_ticket,
        }
        with ApiSession(endpoint=args.endpoint, headers=headers) as session:
            for function in filter(lambda f: f.has_tvm, TESTED):
                print("=== Testing", function.__name__, "with tvm:", function.__doc__)
                function(
                    session=session,
                    args=args,
                )


TESTED = [
    obj
    for obj in globals().values()
    if getattr(obj, 'tested', False)
]


if __name__ == '__main__':
    with warnings.catch_warnings(record=True):
        try:
            main()
        except requests.HTTPError as e:
            print(e.response.headers['X-Served-By'])
            print(e.response.content)
