import datetime
import json
import logging
import itertools
import re
import sys
import time
from collections import defaultdict
from functools import wraps

import base64
import gevent

import cachetools.func
import yp.data_model as data_model

from infra.qyp.account_manager.src.dismissed_vm_owner_worker import DismissedVmOwnerWorker
from infra.qyp.account_manager.src.expired_vm_worker import ExpiredVmWorker
from infra.qyp.account_manager.src import constant
from infra.qyp.account_manager.src import helpers
from infra.qyp.account_manager.src.contrib.abc_client import AbcError
from infra.qyp.account_manager.src.lib.gutils import idle_iter
from infra.qyp.proto_lib import vmset_pb2
from infra.yasm.yasmapi import GolovanRequest
from sepelib.core import config
from yt import yson


log = logging.getLogger('AccountPoller')
SIGNAL_TMPL = 'itype=qemuvm;geo={cluster};prj={pod_id};ctype=prod:unistat-ssh_syn_count_dmmm'
SIGNAL_RE_TMPL = 'itype=qemuvm;geo=(?P<cluster>.*);prj=(?P<pod_id>.*);ctype=prod:unistat-ssh_syn_count_dmmm'
SIGNAL_RE_PTRN = re.compile(SIGNAL_RE_TMPL)

JUPYTER_MASK_TMPL = '^devel-jupyter-cloud-.*|^djc-.*|^testing-jupyter-cloud-.*|^tjc-.*|^jupyter-cloud-.*|^jc-.*'
JUPYTER_MASK_PTRN = re.compile(JUPYTER_MASK_TMPL)
JUPYTER_ABC_ID = 'abc:service:2142'


def timing(f):
    @wraps(f)
    def wrap(*args, **kw):
        f_name = f.__name__
        log.info('run %r', f_name)
        ts = time.time()
        result = f(*args, **kw)
        te = time.time()
        log.info('func:%r took: %2.4f sec', f_name, te-ts)
        return result
    return wrap


class Job:
    def __init__(self, handler, sleep_time, last_run):
        self.handler = handler
        self.sleep_time = sleep_time
        self.last_run_time = last_run

    def set_last_run(self, t):
        """
        :type t: float
        """
        self.last_run_time = t


class AccountPoller(object):
    def __init__(self, ctx):
        self.ctx = ctx
        self.zk_storage = ctx.zk_storage
        self.user_data = defaultdict(list)
        if type(ctx.yp_client_list) is list:
            self.yp_client_list = ctx.yp_client_list
        else:
            self.yp_client_list = [ctx.yp_client_list]

        # TODO: remove if unused
        if type(ctx.pod_ctl_map) is dict:
            self.pod_ctl_list = ctx.pod_ctl_map.values()
        else:
            self.pod_ctl_list = [ctx.pod_ctl_map]

        self.staff_client = ctx.staff_client
        self.abc_client = ctx.abc_client
        self.mail_client = ctx.mail_client
        self.vmproxy_client = ctx.vmproxy_client
        self.jns_client = ctx.jns_client
        self.qdm_client = ctx.qdm_client

        self.overquoting_segments = config.get_value('overquoting_segments', [])
        self.enable_email_send = config.get_value('mail.enable', False)
        self.enable_update_pod = config.get_value('account_poller.enable_update_pods', False)
        self.timeout = config.get_value('account_poller.sleep_secs', constant.SLEEP_SECS)
        self.personal_quota_limit = config.get_value('personal_limit', None)
        self.unused_vm_accounts_to_mark = config.get_value('jobs.unused_vms.accounts_to_mark', [])
        self.unused_vm_nested_accounts_to_mark = config.get_value('jobs.unused_vms.nested_accounts_to_mark', [])
        self.unused_vm_accounts_to_delete = config.get_value('jobs.unused_vms.accounts_to_delete', [])
        self.unused_vm_nested_accounts_to_delete = config.get_value('jobs.unused_vms.nested_accounts_to_delete', [])
        self._services_to_mark = None
        self._services_to_delete = None
        self.jobs_to_run = self.init_jobs()

    def init_jobs(self):
        all_jobs = {
            'vm_outside_sp': self.quota_poller,
            'dismissed_owners': self.vm_owners_poller,
            'overquoting_accounts': self.overquoting_account_poller,
            'resource_usage': self.resource_metrics,
            'unused_vms': self.find_unused_vms,
            'expired_vms': self.expired_vms_poller,
        }
        jobs_to_run = {}
        for key, handler in all_jobs.iteritems():
            if config.get_value('jobs.{}.enable'.format(key), False):
                sleep_time = config.get_value('jobs.{}.sleep_time'.format(key), constant.SLEEP_SECS)
                jobs_to_run[key] = Job(handler, sleep_time, 0)
        return jobs_to_run

    def _push_zookeeper_data(self, json_data, data_id):
        if not self.zk_storage:
            raise NotImplementedError('Not found zookeeper storage instance')
        self.zk_storage.set(json_data, data_id)

    def set_zk_storage(self, zk_storage):
        self.zk_storage = zk_storage

    def _get_dismissed_workers_with_heads(self, workers):
        """
        :type workers: list[ApiStaff.Person]
        :rtype: list[dict[str, str]]
        """
        dismissed_workers = []
        for worker in workers:
            if worker['official']['is_dismissed'] and not worker['official']['is_robot']:
                head_person = None
                for head in worker['department_group']['department']['heads']:
                    if not head['person']['official']['is_dismissed']:
                        head_person = head['person']['login']
                        break
                dismissed_workers.append({
                    'login': str(worker['login']),
                    'new_logins': str(head_person) if head_person is not None else None,
                })
        return dismissed_workers

    def _get_limits_in_account(self, limits, segment):
        """
        Get resource limits in map /spec/resource_limits for segment
        :type limits: map[str, map]
        :type segment: str
        :rtype: list[int]
        """
        ip_limits, mem_limits, cpu_limits, ssd_limits, hdd_limits = 0, 0, 0, 0, 0
        if limits['per_segment'].get(segment) is not None:
            ip_limits = limits['per_segment'][segment].get('internet_address', {'capacity': 0})['capacity']
            cpu_limits = limits['per_segment'][segment]['cpu']['capacity']
            mem_limits = limits['per_segment'][segment]['memory']['capacity']
            hdd_limits = limits['per_segment'][segment]['disk_per_storage_class']['hdd']['capacity']
            ssd_limits = limits['per_segment'][segment]['disk_per_storage_class']['ssd']['capacity']
        return [ip_limits, mem_limits, cpu_limits, ssd_limits, hdd_limits]

    @cachetools.func.ttl_cache(maxsize=1, ttl=1 * 60 * 60)
    def _get_all_groups_in_staff(self):
        """
        Make staff-id groups with qyp-id groups
        Example: staff:department:2900 = 12345
        :rtype: map[str, int]
        """
        spec = {}
        abc_resp = self.abc_client.list_roles_scopes(spec)
        abc_resp = {item['slug']: item['id'] for item in abc_resp['results']}
        staff_data = {
            'department': ('id', 'department.id'),
            'service': ('id', 'service.id'),
            'servicerole': ('id', 'role_scope', 'parent.service.id')
        }
        staff_ids = {}
        for role, fields in staff_data.iteritems():
            spec = {'type': role}
            for page in self.staff_client.list_groups_iter(spec, fields=fields):
                for staff_resp in page['result']:
                    if role == 'department':
                        staff_id = 'staff:department:{}'.format(staff_resp['department']['id'])
                    elif role == 'service':
                        staff_id = 'abc:service:{}'.format(staff_resp['service']['id'])
                    elif role == 'servicerole':
                        service_id = staff_resp['parent']['service']['id']
                        scope_slug = staff_resp['role_scope']
                        scope_id = abc_resp.get(scope_slug)
                        staff_id = 'abc:service-scope:{}:{}'.format(service_id, scope_id)
                    else:
                        continue
                    staff_ids[staff_id] = staff_resp['id']
        return staff_ids

    def _send_golovan_request(self, period, st, et, signals):
        return GolovanRequest('ASEARCH', period, st, et, signals, explicit_fail=True)

    def run(self):
        while True:
            for job_name, job in self.jobs_to_run.items():
                try:
                    if time.time() - job.last_run_time > job.sleep_time:
                        job.handler()
                        job.set_last_run(time.time())
                except Exception as e:
                    log.error('run {} failed: {}'.format(job_name, e))
            gevent.sleep(self.timeout)

    @timing
    def quota_poller(self):
        self.user_data.clear()

        selectors_pod_set = ['/meta/id']
        query_pod_set = '[/labels/deploy_engine]=\'QYP\' AND [/spec/account_id]=\'abc:service:4172\''
        current_vms_dict = defaultdict(list)

        for yp_ctx_client in self.yp_client_list:
            cluster = yp_ctx_client.cluster
            request_pod_set = yp_ctx_client.select_objects(query_pod_set, data_model.OT_POD_SET, selectors_pod_set)
            pod_set_ids = [yson.loads(item.values[0]) for item in request_pod_set.results]

            if len(pod_set_ids) == 0:
                continue

            selectors_pod = ['/annotations/owners/author',
                             '/meta']
            query_pod = '[/meta/pod_set_id] in ({})'.format(','.join(helpers.quoted(pod_set_ids)))
            request_pod = yp_ctx_client.select_objects(query_pod, data_model.OT_POD, selectors_pod)

            users_in_host = []
            # item.values[0] -> author, item.values[1] -> meta
            for item in request_pod.results:
                author = yson.loads(item.values[0])
                if author != '#':
                    meta = yson.loads(item.values[1])
                    meta['author'] = author
                    self.user_data[author].append(meta)
                    users_in_host.append(author)

            request_permissions = yp_ctx_client.check_object_permissions_few(constant.QYP_PERSONAL_ID,
                                                                             data_model.OT_ACCOUNT,
                                                                             users_in_host, data_model.ACA_USE)

            for user, request in request_permissions:
                if request == data_model.ACA_DENY:
                    current_vms_dict[cluster.get('cluster_name')].extend(self.user_data[user])

        self._push_zookeeper_data(current_vms_dict, 'vm_outside_sp')

    @timing
    def vm_owners_poller(self):
        DismissedVmOwnerWorker(zk_storage=self.zk_storage, yp_client_list=self.yp_client_list).run()

    @timing
    def expired_vms_poller(self):
        ExpiredVmWorker(self.yp_client_list, self.vmproxy_client, self.jns_client, self.qdm_client).run()

    @timing
    def overquoting_account_poller(self):
        current_overquoting_accounts = defaultdict(list)
        for yp_ctx_client in self.yp_client_list:
            cluster = yp_ctx_client.cluster

            selectors = ['/meta/id',
                         '/status/resource_usage',
                         '/spec/resource_limits']
            query = None
            accounts = yp_ctx_client.list_accounts(query, selectors=selectors)

            for account in accounts.results:
                service_id, usage, limits = [yson.loads(value) for value in account.values]
                if usage.get('per_segment') is None or service_id == 'tmp':
                    continue
                for key, segment in usage['per_segment'].items():
                    if key not in self.overquoting_segments:
                        continue

                    ip_limits, mem_limits, cpu_limits, ssd_limits, hdd_limits = self._get_limits_in_account(limits, key)

                    current_overquoting = {}
                    if segment.get('internet_address') is not None:
                        if ip_limits < segment['internet_address']['capacity']:
                            current_overquoting['internet_address'] = segment['internet_address']['capacity'] - ip_limits

                    if mem_limits < segment['memory']['capacity']:
                        current_overquoting['memory'] = segment['memory']['capacity'] - mem_limits

                    if cpu_limits < segment['cpu']['capacity']:
                        current_overquoting['cpu'] = segment['cpu']['capacity'] - cpu_limits

                    if segment.get('disk_per_storage_class') is not None:
                        if segment['disk_per_storage_class'].get('hdd') is not None:
                            if hdd_limits < segment['disk_per_storage_class']['hdd']['capacity']:
                                current_overquoting['hdd'] = segment['disk_per_storage_class']['hdd']['capacity'] - hdd_limits

                        if segment['disk_per_storage_class'].get('ssd') is not None:
                            if ssd_limits < segment['disk_per_storage_class']['ssd']['capacity']:
                                current_overquoting['ssd'] = segment['disk_per_storage_class']['ssd']['capacity'] - ssd_limits

                    if len(current_overquoting) > 0:
                        current_overquoting_accounts[cluster.get('cluster_name')].append({
                            'id': service_id,
                            'segment': key,
                            'limits': limits['per_segment'][key],
                            'usage': usage['per_segment'][key],
                            'overquoting': current_overquoting
                        })
        self._push_zookeeper_data(current_overquoting_accounts, 'overquoting_accounts')

    @timing
    def resource_metrics(self):
        selectors_accounts = ['/meta/id', '/spec/resource_limits/per_segment']
        selectors_node_segments = ['/status/schedulable_resources',
                                   '/status/total_resources']
        signals = []

        nested_dict = lambda: defaultdict(nested_dict)
        resource_usage_stats = nested_dict()

        for yp_ctx_client in self.yp_client_list:
            cluster_name = yp_ctx_client.cluster.get('cluster_name')
            unistat = dict()
            unistat['granted'] = helpers.create_dict_usage_metrics()
            accounts = yp_ctx_client.list_accounts(None, selectors=selectors_accounts)
            accounts = [[yson.loads(i) for i in val.values] for val in accounts.results]
            for account in accounts:
                if bool(account[1]) and account[0] not in constant.REMOVE_ACCOUNT_IDS_METRICS:
                    for segment, data in account[1].iteritems():
                        cpu = data.get('cpu', {}).get('capacity', 0)
                        mem = data.get('memory', {}).get('capacity', 0)
                        ssd = data.get('disk_per_storage_class', {}).get('ssd', {}).get('capacity', 0)
                        hdd = data.get('disk_per_storage_class', {}).get('hdd', {}).get('capacity', 0)
                        if segment == constant.DEV_SEGMENT_ID:
                            unistat['granted']['cpu'] += cpu
                            unistat['granted']['mem'] += mem
                            unistat['granted']['ssd'] += ssd
                            unistat['granted']['hdd'] += hdd
                        resource_usage_stats[segment][cluster_name]['cpu'][account[0]] = cpu
                        resource_usage_stats[segment][cluster_name]['mem'][account[0]] = mem
                        resource_usage_stats[segment][cluster_name]['ssd'][account[0]] = ssd
                        resource_usage_stats[segment][cluster_name]['hdd'][account[0]] = hdd

            node_segments = yp_ctx_client.get_node_segment('dev', selectors_node_segments)
            segments = ['scheduled', 'total']
            node_segments = [[yson.loads(val[0]), val[1]] for val in zip(node_segments.result.values, segments)]

            for node_segment, resource_name in node_segments:
                unistat[resource_name] = helpers.create_dict_usage_metrics()
                unistat[resource_name]['cpu'] += node_segment.get('cpu', {}).get('capacity', 0)
                unistat[resource_name]['mem'] += node_segment.get('memory', {}).get('capacity', 0)
                unistat[resource_name]['ssd'] += node_segment.get('disk_per_storage_class', {}).get('ssd', {}).get(
                    'capacity', 0)
                unistat[resource_name]['hdd'] += node_segment.get('disk_per_storage_class', {}).get('hdd', {}).get(
                    'capacity', 0)

            for resource in ('cpu', 'mem', 'ssd', 'hdd'):
                signals += [['{}_{}_scheduled_axxv'.format(cluster_name, resource), unistat['scheduled'][resource]]]
                signals += [['{}_{}_total_axxv'.format(cluster_name, resource), unistat['total'][resource]]]
                signals += [['{}_{}_granted_axxv'.format(cluster_name, resource), unistat['granted'][resource]]]

        for segment, cluster_data in resource_usage_stats.iteritems():
            for cluster in cluster_data.iterkeys():
                for resource in ('cpu', 'mem', 'ssd', 'hdd'):
                    resource_data = [[key, val] for key, val in cluster_data[cluster][resource].iteritems()]
                    resource_data.sort(key=lambda x: x[1], reverse=True)
                    resource_usage_stats[segment][cluster][resource] = resource_data
        self._push_zookeeper_data(signals, 'resource_usage_metrics')
        self._push_zookeeper_data(resource_usage_stats, 'resource_usage_statistics')

    def _get_vm_os_old(self, resources_spec):
        try:
            idx_config = [idx for idx, item in enumerate(resources_spec) if item['key'] == 'vm.config'][0]
        except IndexError:
            return None
        if 'resource' in resources_spec[idx_config]['value']:
            url_config = resources_spec[idx_config]['value']['resource']['urls'][0]
        elif 'dynamicResource' in resources_spec[idx_config]['value']:
            url_config = resources_spec[idx_config]['value']['dynamicResource']['urls'][0]
        else:
            return None
        url_config = url_config.split('data:text/plain;charset=utf-8;base64,')[1]
        decoded_dict = json.loads(base64.b64decode(url_config))
        return decoded_dict.get('type') or 'LINUX'

    @staticmethod
    def split_iterable_into_batches(iterable, batch_size=50):
        it = iter(iterable)
        batch = list(itertools.islice(it, batch_size))
        while len(batch) > 0:
            yield batch
            batch = list(itertools.islice(it, batch_size))

    def _update_vm_usage_labels(self, vm_ids_by_cluster, is_unused):
        """
        :type vm_ids_by_cluster: dict[str, set]
        :type is_unused: bool
        :rtype: None
        """
        update_field = {'/labels/qyp_{}'.format(constant.UNUSED_VM_LABEL_NAME): str(is_unused).lower()}
        for yp_client in self.yp_client_list:
            cluster = yp_client.cluster.get('cluster_name').lower()
            if cluster not in vm_ids_by_cluster:
                continue

            for ids in self.split_iterable_into_batches(vm_ids_by_cluster[cluster]):
                updates = [update_field for _ in range(len(ids))]
                try:
                    yp_client.update_objects(object_ids=ids, object_type=data_model.OT_POD_SET, set_updates=updates)
                except Exception as e:
                    log.warning('Unable to update pods: {}'.format(e))

    def _get_child_services(self, abc_services):
        """
        :type abc_services: list[str]
        :rtype: list[str]
        """
        res = []
        for abc_service in abc_services:
            service_id = self.get_abc_service_id(abc_service)
            if service_id == constant.TEMPORARY_ACCOUNT_ID:
                res.append(service_id)
                continue
            else:
                res.append('abc:service:{}'.format(service_id))
            try:
                results = self.abc_client.list_services(
                    spec={
                        'parent__with_descendants': service_id,
                        'fields': ['id'],
                        'state__in': 'develop,supported'
                    }
                )
                for r in results:
                    child_id = r.get('id')
                    if child_id:
                        res.append('abc:service:{}'.format(child_id))
            except AbcError as e:
                log.warning('Unable to get child services for abc_service: {}, err: {}'.format(service_id, e))
        return res

    @staticmethod
    def get_abc_service_id(abc_service):
        """
        :type abc_service: str
        :rtype: str
        """
        if abc_service == constant.TEMPORARY_ACCOUNT_ID:
            return abc_service
        service_id = abc_service.split(':')[-1]
        return service_id

    @property
    def services_to_mark(self):
        if self._services_to_mark is None:
            services = set(self.unused_vm_accounts_to_mark)
            services.update(self._get_child_services(self.unused_vm_nested_accounts_to_mark))
            self._services_to_mark = services
        return self._services_to_mark

    @property
    def services_to_delete(self):
        if self._services_to_delete is None:
            services = set(self.unused_vm_accounts_to_delete)
            services.update(self._get_child_services(self.unused_vm_nested_accounts_to_delete))
            self._services_to_delete = services
        return self._services_to_delete

    @timing
    def find_unused_vms(self):
        count_days = constant.DEFAULT_DAYS
        signals = []
        nested_dict = lambda: defaultdict(nested_dict)
        global_info_pod = nested_dict()
        query = '[/labels/deploy_engine]=\'QYP\''
        selectors = [
            '/meta/id',
            '/meta/creation_time',
            '/annotations/qyp_vm_spec',
        ]

        for yp_ctx_client in self.yp_client_list:
            cluster = yp_ctx_client.cluster.get('cluster_name').lower()

            pod_ids = yp_ctx_client.list_pods(query, selectors=selectors)
            for pod_result in idle_iter(pod_ids.results):
                pod_id, creation_time, vm_spec_raw = [yson.loads(val) for val in pod_result.values]
                created_days_ago = helpers.get_days_delta(creation_time/10**6)  # YP creation time in microseconds
                if not vm_spec_raw:
                    # old vmagent appears only in default segment
                    continue
                vm_pb = vmset_pb2.VM.FromString(vm_spec_raw)
                if vm_pb.spec.qemu.node_segment not in ('dev', 'gpu-dev'):
                    continue
                if vm_pb.spec.qemu.vm_type == vmset_pb2.VMType.WINDOWS:
                    continue
                if vm_pb.spec.account_id == JUPYTER_ABC_ID or JUPYTER_MASK_PTRN.findall(pod_id):
                    # Filter jupyter vms
                    continue
                if created_days_ago < count_days:
                    continue
                if vm_pb.spec.account_id in self.services_to_mark:
                    global_info_pod[cluster][pod_id]['qnotifier_pod_data'] = {
                        'id': pod_id,
                        'owners': vm_pb.meta.auth.owners,
                        'created_days_ago': created_days_ago,
                        'account_id': vm_pb.spec.account_id,
                        'cluster': cluster
                    }
                    signals.append(SIGNAL_TMPL.format(cluster=cluster, pod_id=pod_id))

        self.process_unused_vms(global_info_pod, signals, count_days)

    def process_unused_vms(self, global_info_pod, signals, count_days):
        job_config = config.get_value('jobs.unused_vms', {})
        et, st = helpers.get_et_st(constant.PERIOD, count_days)
        nested_dict = lambda: defaultdict(nested_dict)
        result = nested_dict()

        """
        TABLE
        [
            {'pod_id': str, 'cluster': str, 'state': int, 'last_timestamp': int, 'unused_days': int},
        ]
        STATE
        1 - VM inactive 31+ days, does not exceed 45 days
        2 - VM inactive 45+ days, does not exceed 58 days
        3 - VM inactive 58+ days, does not exceed 59 days
        4 - VM send request for backup and need check backup
        5 - VM need dealloc
        """
        bad_pod_ids = set()
        for step in idle_iter(range(0, len(signals), 200)):
            gevent.sleep(2)
            for timestamp, values in self._send_golovan_request(constant.PERIOD, st, et, signals[step: step + 200]):
                for key, val in values.iteritems():
                    cluster, pod_id = SIGNAL_RE_PTRN.search(key).groups()
                    if val is None:
                        bad_pod_ids.add((cluster, pod_id))
                        val = 0
                    result[cluster][pod_id]['summary'] = result[cluster][pod_id].get('summary', 0) + val
                    if 'unused_days' not in result[cluster][pod_id]:
                        result[cluster][pod_id]['unused_days'] = 0
                    if val:
                        result[cluster][pod_id]['unused_days'] = 0
                    else:
                        result[cluster][pod_id]['unused_days'] = min(
                            global_info_pod[cluster][pod_id]['qnotifier_pod_data']['created_days_ago'],
                            result[cluster][pod_id]['unused_days'] + 1
                        )

        for cluster, pod_id in bad_pod_ids:
            del result[cluster][pod_id]

        from_zk_storage = self.zk_storage.get('table_unused_vms') or []
        vm_ids_in_table = set()
        watched_vms_count = 0
        for row in from_zk_storage:
            cluster = row['cluster']
            pod_id = row['pod_id']
            vm_ids_in_table.add('{}#{}'.format(cluster, pod_id))
            if cluster in result and pod_id in result[cluster]:
                watched_vms_count += 1

        vms_move_state_1, vms_move_state_2, vms_move_state_3, vms_move_state_4 = [], [], [], []
        vms_dealloc, vms_without_changes, vms_check_backup = [], [], {}

        pods_from_notification = []
        current_timestamp = int(time.time())

        vms_limit = job_config.get('vms_limit', None)
        for cluster in result.iterkeys():
            for pod_id in idle_iter(result[cluster].iterkeys()):
                if vms_limit is not None and watched_vms_count >= vms_limit:
                    break
                if result[cluster][pod_id]['summary'] != 0:
                    # VM is actually in use or was created earlier than a month
                    continue
                if '{}#{}'.format(cluster, pod_id) in vm_ids_in_table:
                    # We have already found this VM on previous steps
                    continue
                vms_move_state_1.append({
                    'pod_id': pod_id,
                    'cluster': cluster,
                    'state': 1,
                    'last_timestamp': current_timestamp,
                    'unused_days': count_days,
                })
                pods_from_notification.append({
                    'pod_id': pod_id,
                    'cluster': cluster,
                    'state': 1,
                    'last_timestamp': current_timestamp
                })
                watched_vms_count += 1
        log.info('Watch over {} vms'.format(watched_vms_count))

        for row in idle_iter(from_zk_storage):
            if row['cluster'] not in result or row['pod_id'] not in result[row['cluster']]:
                continue
            if result[row['cluster']][row['pod_id']]['summary'] == 0:
                days_on_last_update = (current_timestamp - row['last_timestamp']) / constant.PERIOD
                if row['state'] == 1:
                    row['unused_days'] = count_days + days_on_last_update
                    if days_on_last_update >= constant.COUNT_DAYS_MOVE_STATE_2:
                        row['state'] = 2
                        row['last_timestamp'] = current_timestamp
                        vms_move_state_2.append(row)
                        pods_from_notification.append(row)
                    else:
                        vms_without_changes.append(row)
                elif row['state'] == 2:
                    row['unused_days'] = count_days + constant.COUNT_DAYS_MOVE_STATE_2 + days_on_last_update
                    if days_on_last_update >= constant.COUNT_DAYS_MOVE_STATE_3:
                        row['state'] = 3
                        row['last_timestamp'] = current_timestamp
                        vms_move_state_3.append(row)
                        pods_from_notification.append(row)
                    else:
                        vms_without_changes.append(row)
                elif row['state'] == 3:
                    row['unused_days'] = count_days + constant.COUNT_DAYS_MOVE_STATE_2 + constant.COUNT_DAYS_MOVE_STATE_3 + days_on_last_update
                    if days_on_last_update >= constant.COUNT_DAYS_MOVE_STATE_4:
                        row['state'] = 4
                        row['last_timestamp'] = current_timestamp
                        vms_move_state_4.append(row)
                    else:
                        vms_without_changes.append(row)
                elif row['state'] == 4:
                    vms_check_backup[row['pod_id']] = row

        if vms_check_backup:
            for row in vms_check_backup.itervalues():
                cluster, pod_id = row['cluster'], row['pod_id']
                backups_list = self.vmproxy_client.list_backups(cluster, pod_id)
                if backups_list and backups_list[-1].status.state == vmset_pb2.BackupStatus.COMPLETED:
                    row['state'] = 5
                    row['backup_url'] = backups_list[-1].status.url
                    vms_dealloc.append(row)
                    pods_from_notification.append(row)
                elif not backups_list or backups_list[-1].status.state != vmset_pb2.BackupStatus.IN_PROGRESS:
                    # Backup failed, run again
                    row['unused_days'] = 59
                    vms_move_state_4.append(row)

        if job_config.get('enable_update', False):
            for row in idle_iter(vms_dealloc):
                cluster, pod_id = row['cluster'], row['pod_id']
                if global_info_pod[cluster][pod_id]['qnotifier_pod_data']['account_id'] in self.services_to_delete:
                    self.vmproxy_client.deallocate(cluster, pod_id)
            for row in idle_iter(vms_move_state_4):
                cluster, pod_id = row['cluster'], row['pod_id']
                if global_info_pod[cluster][pod_id]['qnotifier_pod_data']['account_id'] in self.services_to_delete:
                    self.vmproxy_client.backup(cluster, pod_id)

        if job_config.get('enable_notifications', False):
            for row in idle_iter(pods_from_notification):
                cluster = row['cluster']
                pod_id = row['pod_id']
                if global_info_pod[cluster][pod_id]['qnotifier_pod_data']['account_id'] in self.services_to_delete:
                    self._send_mails(
                        cluster=cluster,
                        pod_info=global_info_pod[cluster][pod_id]['qnotifier_pod_data'],
                        step=row['state'],
                        backup_url=row.get('backup_url', None)
                    )

        data_to_zookeeper = []
        data_to_zookeeper.extend(vms_move_state_1)
        data_to_zookeeper.extend(vms_move_state_2)
        data_to_zookeeper.extend(vms_move_state_3)
        data_to_zookeeper.extend(vms_move_state_4)
        data_to_zookeeper.extend(vms_without_changes)

        log.info('Size of zk objects: unused_days - {}, states - {}'.format(
            sys.getsizeof(json.dumps(result)),
            sys.getsizeof(json.dumps(data_to_zookeeper))
        ))
        self._push_zookeeper_data(result, 'table_unused_vms_all')
        self._push_zookeeper_data(data_to_zookeeper, 'table_unused_vms')
        self.mark_unused_vms(data_to_zookeeper, vm_ids_in_table)

    def _send_mails(self, cluster, pod_info, step, backup_url=None):
        pod_id = pod_info['id']
        days_left = 0
        if step == 1:
            template = 'unused_vm_warning'
            days_left = constant.COUNT_DAYS_MOVE_STATE_1
        elif step == 2:
            template = 'unused_vm_warning'
            days_left = constant.COUNT_DAYS_MOVE_STATE_2
        elif step == 3:
            template = 'unused_vm_last_warning'
            days_left = constant.COUNT_DAYS_MOVE_STATE_4
        elif step == 5:
            template = 'unused_vm_removed'
        else:
            # Do not need to send anything
            return True

        logins = list(pod_info['owners'].logins)
        params = dict(
            vm_id=pod_id,
            qyp_link=constant.QYP_LINK_TMPL.format(cluster, pod_id),
            docs_link=constant.DOCS_LINK_FOR_UNUSED,
            backup_url=backup_url,
            owners=', '.join(logins),
            date=(datetime.datetime.now() + datetime.timedelta(days=days_left)).strftime('%d.%m.%Y'),
            days_left=str(days_left)
        )
        logins += ['frolstas']  # while testing
        result = self.jns_client.send(logins, template, params)
        if result:
            log.info('Notifications for %s sent successfully', pod_id)
        else:
            log.warning('Unable to notify %s', pod_id)

    def mark_unused_vms(self, to_mark_vms, marked_vms):
        """
        :type to_mark_vms: list
        :type marked_vms: set
        """
        to_mark_vms_table = set()

        for row in to_mark_vms:
            cluster, pod_id = row['cluster'], row['pod_id']
            to_mark_vms_table.add('{}#{}'.format(cluster, pod_id))

        to_mark = defaultdict(set)
        to_unmark = defaultdict(set)

        for row in to_mark_vms_table.difference(marked_vms):
            cluster, pod_id = row.split("#")
            to_mark[cluster].add(pod_id)

        for row in marked_vms.difference(to_mark_vms_table):
            cluster, pod_id = row.split("#")
            to_unmark[cluster].add(pod_id)
        self._update_vm_usage_labels(to_mark, True)
        self._update_vm_usage_labels(to_unmark, False)
