from collections import defaultdict
import cachetools.func
import logging
from gevent.pool import Pool

import yp.data_model as data_model
from yt import yson
from yt.packages.requests import exceptions as yt_request_exceptions
import yt_yson_bindings

from infra.qyp.proto_lib import vmset_pb2, accounts_api_pb2
from infra.qyp.account_manager.src import errors
from infra.qyp.account_manager.src.model import abc_roles


QYP_SEGMENTS = ['dev', 'default', 'gpu-dev']
log = logging.getLogger(__name__)


@cachetools.func.ttl_cache(maxsize=5, ttl=3 * 60 * 60)
def get_vms_by_author(pod_ctl, account_id):
    """
    caches pods for login
    :type pod_ctl: account_manager.src.model.PodController
    :type account_id: str
    :rtype: dict[str, dict[str, List[str]]]
    """
    res = defaultdict(dict)
    if not account_id:
        return res
    # {account-id: {segment: [pods]}}
    pod_sets_by_segment = defaultdict(list)
    q = '[/meta/account_id] = "{}"'.format(account_id)
    rsp = pod_ctl.yp_client.select_objects(q, data_model.OT_POD_SET, selectors=['/meta/id', '/spec/node_segment_id'])
    for v in rsp.results:
        pod_set_id, segment = yson.loads(v.values[0]), yson.loads(v.values[1])
        pod_sets_by_segment[segment].append(pod_set_id)
    # select pods
    for segment, pod_set_ids in pod_sets_by_segment.iteritems():
        pods = pod_ctl.get_pods(pod_set_ids, ['/meta/id', '/annotations/owners/author'])
        for r in pods.results:
            pod_id, author = yson.loads(r.values[0]), yson.loads(r.values[1])
            if segment in res[author]:
                res[author][segment].append(pod_id)
            else:
                res[author][segment] = [pod_id]
    return res


def set_personal_limits(limits, personal_quota):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type personal_quota: dict
    """
    per_segment_limits = limits.per_segment[personal_quota['segment']]
    per_segment_limits.mem = personal_quota['mem']
    per_segment_limits.cpu = personal_quota['cpu']
    for disk in personal_quota['disk']:
        per_segment_limits.disk_per_storage[disk['storage']] = disk['capacity']
        per_segment_limits.io_guarantees_per_storage[disk['storage']] = disk['bandwidth_guarantee']
    per_segment_limits.internet_address = personal_quota['internet_address']
    for gpu in personal_quota.get('gpu_per_model', []):
        per_segment_limits.gpu_per_model[gpu['model']] = gpu['limit']


def set_personal_usage(pod_ctl, usage, vms):
    """
    :type pod_ctl: account_manager.src.model.PodController
    :type usage: vmset_pb2.ResourceTotals
    :type vms: dict[str, List[str]]
    :rtype: None
    """
    for segment in vms:
        pods = pod_ctl.get_pods(vms[segment], selectors=['/spec'])
        for p in pods.results:
            spec = yt_yson_bindings.loads_proto(p.values[0], data_model.TPodSpec)
            per_segment_usage = usage.per_segment[segment]
            per_segment_usage.cpu += spec.resource_requests.vcpu_guarantee
            per_segment_usage.mem += spec.resource_requests.memory_guarantee
            for volume in spec.disk_volume_requests:
                per_segment_usage.disk_per_storage[volume.storage_class] += volume.quota_policy.capacity
                per_segment_usage.io_guarantees_per_storage[volume.storage_class] += volume.quota_policy.bandwidth_guarantee
            for ip6 in spec.ip6_address_requests:
                if ip6.enable_internet:
                    per_segment_usage.internet_address += 1
                    break
            for gpu in spec.gpu_requests:
                per_segment_usage.gpu_per_model[gpu.model] += 1


def cast_yp_resources(pb, yp_resources):
    """
    :type pb: vmset_pb2.ResourceInfo
    :type yp_resources: data_model_pb2.TPerSegmentResourceTotals
    """
    pb.mem = yp_resources.memory.capacity
    pb.cpu = yp_resources.cpu.capacity
    for key, value in yp_resources.disk_per_storage_class.iteritems():
        pb.disk_per_storage[key] = value.capacity
        pb.io_guarantees_per_storage[key] = value.bandwidth
    pb.internet_address = yp_resources.internet_address.capacity
    for key, value in yp_resources.gpu_per_model.iteritems():
        pb.gpu_per_model[key] = value.capacity
    pb.network_guarantee = yp_resources.network.bandwidth


def set_limits(limits, account, segment=None):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.spec.resource_limits.per_segment:
        cast_yp_resources(limits.per_segment[segment], account.spec.resource_limits.per_segment[segment])
    else:
        for segment_, segment_limits in account.spec.resource_limits.per_segment.iteritems():
            if segment_ not in QYP_SEGMENTS:
                continue
            cast_yp_resources(limits.per_segment[segment_], segment_limits)


def set_usage(usage, account, segment=None):
    """
    :type usage: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.status.resource_usage.per_segment:
        cast_yp_resources(usage.per_segment[segment], account.status.resource_usage.per_segment[segment])
    else:
        for segment_, segment_usage in account.status.resource_usage.per_segment.iteritems():
            if segment_ not in QYP_SEGMENTS:
                continue
            cast_yp_resources(usage.per_segment[segment_], segment_usage)


@cachetools.func.ttl_cache(maxsize=5, ttl=3 * 60 * 60)
def get_search_members(pod_ctl):
    """
    :rtype: frozenset
    """
    # All search members are included in QYP personal account for now
    rsp = pod_ctl.yp_client.get_group('abc:service:4172', selectors=['/meta/id', '/spec/members'])
    return frozenset(yson.loads(rsp.result.values[1]))


@cachetools.func.ttl_cache(maxsize=5, ttl=5 * 60)
def get_yp_superusers(pod_ctl):
    """
    :rtype: frozenset
    """
    group_rsp = pod_ctl.yp_client.get_group('superusers')
    members = yson.loads(group_rsp.result.values[0]).get('spec', {}).get('members', [])
    return frozenset(members)


def run(pod_ctl, abc_client, login, account_id, segment, use_cache, personal_quotas_dict):
    """
    :type pod_ctl: account_manager.src.model.PodController
    :type abc_client: AbcClient
    :type login: str
    :type account_id: str
    :type segment: str
    :type use_cache: bool
    :type personal_quotas_dict: dict
    :rtype: List[vmset_pb2.Account]
    """
    if account_id:
        if not pod_ctl.check_use_account_permission(account_id, login, use_cache, abc_client):
            raise errors.AuthorizationError('{} must be a member of a {} account'.format(login, account_id))
        account_ids = [account_id]
    else:
        if login in get_yp_superusers(pod_ctl):
            service_ids = abc_roles.list_user_services_from_abc(abc_client, login)
            account_ids = ['abc:service:{}'.format(service_id) for service_id in service_ids]
        else:
            a = pod_ctl.list_account_ids_by_logins([login])
            account_ids = []
            for ac in a.values():
                account_ids.extend(ac)
        account_ids = abc_roles.filter_accounts_by_roles(abc_client, account_ids, login, use_cache)
        if not account_ids:
            return []

    accounts = pod_ctl.get_accounts(account_ids)
    groups = pod_ctl.yp_client.get_objects(account_ids, data_model.OT_GROUP, selectors=['/meta/id', '/spec/members'])
    members_dict = {}
    for item in groups.subresponses:
        if len(item.result.values) == 2:
            members_dict[yson.loads(item.result.values[0])] = item.result.values[1]
    result = []
    personal_account_ids = list(set(account_ids).intersection(set(personal_quotas_dict)))
    vm_ids = defaultdict(dict)   # {acc: {segment: [vms]} }
    if personal_account_ids:
        for acc in personal_account_ids:
            vms_by_author = get_vms_by_author(pod_ctl, acc)  # - returns cached pod_ids by segment for login
            for segment in vms_by_author[login]:
                if segment in vm_ids[acc]:
                    vm_ids[acc][segment].extend(vms_by_author[login][segment])
                else:
                    vm_ids[acc][segment] = vms_by_author[login][segment]
    for account in accounts:
        acc = vmset_pb2.Account()
        acc.id = account.meta.id
        set_limits(acc.limits, account, segment)
        set_usage(acc.usage, account, segment)
        if acc.id in members_dict:
            service_id = acc.id.split('abc:service:')[-1]
            members = abc_roles.get_service_qyp_persons(abc_client, [acc.id], use_cache)[service_id] or list(yson.loads(members_dict[acc.id]))
            search_members = get_search_members(pod_ctl)
            acc.members_count = sum(int(member in search_members) for member in members)
        if account.meta.id in personal_quotas_dict:
            acc.type = vmset_pb2.Account.PERSONAL
            set_personal_limits(acc.personal.limits, personal_quotas_dict[account.meta.id])
            set_personal_usage(pod_ctl, acc.personal.usage, vm_ids[account.meta.id])
        else:
            acc.type = vmset_pb2.Account.SERVICE
        result.append(acc)
    return result


def run_for_clusters(req, ctx):
    """
    :type req: accounts_api_pb2.ListUserAccountsRequest
    :type ctx: infra.qyp.account_manager.src.main.Ctx
    :rtype: dict
    """
    res = {}
    use_cache = req.consistency == accounts_api_pb2.WEAK

    def task(c):
        try:
            acs = run(
                ctx.pod_ctl_map[c], ctx.abc_client,
                req.login, req.account_id, req.segment, use_cache, ctx.personal_quota
            )
            res[c] = acs
        except yt_request_exceptions.RequestException as e:
            log.warning('Exception counting quota from %s: %s', cluster, e)

    clusters = []
    for cluster in ctx.pod_ctl_map:
        if req.clusters and cluster not in req.clusters:
            continue
        clusters.append(cluster)

    p = Pool(size=len(clusters))
    p.map(task, clusters)

    return res
