# coding: utf-8
from collections import defaultdict

import copy
from infra.swatlib.httpclient import HttpClient, HttpClientException


class AbcError(HttpClientException):
    pass


class AbcClient(object):
    ABC_API_URL = 'https://abc-back.yandex-team.ru/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_RESPONSE_LIMIT = 100
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(abc_url=d.get('api_url'),
                   oauth_token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'))

    def __init__(self, abc_url=None, oauth_token=None, req_timeout=None, verify_ssl=None, max_retries=None):
        abc_url = abc_url or self.ABC_API_URL
        token = oauth_token
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='abc',
                                  exc_cls=AbcError,
                                  base_url=abc_url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)

    def list_service_members(self, service_id, spec={}, remove_robot=False):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service members.

        :type service_id: int
        :type spec: dict[str, str], default None
        :type remove_robot: bool, default False
        :rtype: list[str]
        """
        result = []
        full_spec = {'service': service_id, 'fields': 'person'}
        full_spec.update(spec)
        for resp in self.list_members_iter(spec=full_spec):
            for item in resp['results']:
                if remove_robot and item['person']['is_robot']:
                    continue
                result.append(str(item['person']['login']))
        return result

    def is_service_member(self, login, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns True if user is a service member, False otherwise

        :type service_id: int
        :type login: str
        :rtype: bool
        """
        for resp in self.list_members_iter(spec={'person__login': login, 'fields': 'service'}):
            for item in resp['results']:
                if item['service']['id'] == service_id:
                    return True
        return False

    def list_user_services(self, login):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?person__login=login

        Returns a mapping service_id:list_of_role_ids for given user

        :type login: str
        :rtype: dict[str, list[str]]
        """
        all_services = defaultdict(list)
        for resp in self.list_members_iter(spec={'person__login': login, 'fields': 'service,role'}):
            for item in resp['results']:
                all_services[str(item['service']['id'])].append(str(item['role']['id']))
        return all_services

    def list_service_member_roles(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/members/?service=service_id

        Returns a list of service member roles.

        :type service_id: int
        :rtype: list[str]
        """
        result = []
        for resp in self.list_members_iter(spec={'service': service_id, 'fields': 'role'}):
            for item in resp['results']:
                result.append(str(item['role']['scope']['slug']))
        return result

    def get_service_slug(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service slug

        :type service_id: int
        :rtype: str
        """
        return str(self.get_service(service_id)['slug'])

    def get_service(self, service_id):
        """
        https://abc-back.yandex-team.ru/api/v4/services/service_id/

        Returns the service by id

        :type service_id: int
        :rtype: dict
        """
        return self._client.get('/api/v4/services/{}/'.format(service_id))

    def multicomplete(self, spec):
        return self._client.get('/multic/', params=spec)

    def list_services(self, spec):
        results = []
        for rsp in self.list_services_iter(spec):
            res = rsp.get('results', [])
            results.extend(res)
        return results

    def list_services_iter(self, spec):
        resp = self._client.get('/api/v4/services/', params=self._set_response_limit(spec))
        yield resp
        next_url = resp.get('next')
        while next_url:
            resp = self._client.get(next_url, params=spec)
            yield resp
            next_url = resp.get('next')

    def list_roles_scopes(self, spec):
        return self._client.get('/api/v4/roles/scopes/', params=self._set_response_limit(spec))

    def list_members(self, spec):
        return self._client.get('/api/v4/services/members/', params=self._set_response_limit(spec))

    def list_members_iter(self, spec):
        spec = self._set_response_limit(spec)
        resp = self.list_members(spec)
        yield resp
        next_url = resp.get('next')
        while next_url:
            resp = self._client.get(next_url, params=spec)
            yield resp
            next_url = resp.get('next')

    def _set_response_limit(self, spec):
        if 'page_size' in spec:
            return spec
        new_spec = copy.deepcopy(spec)
        new_spec['page_size'] = self.DEFAULT_RESPONSE_LIMIT
        return new_spec
