import six
from infra.swatlib.httpclient import HttpClient, HttpClientException


class StaffError(HttpClientException):
    pass


class StaffClient(object):
    STAFF_API_URL = 'https://staff-api.yandex-team.ru/v3/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_RESPONSE_LIMIT = 200
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(staff_url=d.get('api_url'),
                   oauth_token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   limit=d.get('response_entities_limit'),
                   max_retries=d.get('max_retries'))

    def __init__(self, staff_url=None, oauth_token=None, req_timeout=None, verify_ssl=None,
                 limit=None, max_retries=None):
        staff_url = staff_url or self.STAFF_API_URL
        token = oauth_token
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = HttpClient(client_name='staff',
                                  exc_cls=StaffError,
                                  base_url=staff_url,
                                  req_timeout=req_timeout,
                                  token=token,
                                  verify=verify_ssl,
                                  max_retries=max_retries)
        self._limit = self.DEFAULT_RESPONSE_LIMIT if limit is None else limit

    def _list(self, rel_url, spec, fields, one, limit):
        params = dict((key, value if not isinstance(value, list) else ','.join(map(str, value)))
                      for key, value in six.iteritems(spec))
        params['_fields'] = ','.join(fields)
        if one:
            params['_one'] = '1'
        if limit is None:
            params['_limit'] = self._limit
        else:
            params['_limit'] = limit
        return self._client.get(rel_url, params=params)

    def list_persons(self, spec, fields=('id', 'name'), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/persons?_doc=1
        """
        return self._list('persons', spec, fields, one, limit)

    def list_groups(self, spec, fields=('id', 'name'), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/groups?_doc=1
        """
        return self._list('groups', spec, fields, one, limit)

    def list_persons_subrequests(self, spec_name, spec_data, fields=('id', 'name'), subrequst_step=None):
        if subrequst_step is None:
            return self.list_persons(spec={spec_name: ','.join(spec_data)}, fields=fields)
        request_data = []
        count_logins = len(spec_data)
        current_step = 0
        while current_step < count_logins:
            staff_spec = {'login': ','.join(spec_data[current_step:current_step + subrequst_step])}
            request_data.extend(self.list_persons(spec=staff_spec, fields=fields, limit=subrequst_step)['result'])
            current_step += subrequst_step
        return request_data

    def list_groups_iter(self, spec, fields=('id', 'name'), limit=1000):
        resp = self.list_groups(spec, fields=fields, limit=limit)
        yield resp
        total_pages = int(resp['pages'])
        if total_pages < 2:
            return
        for page_index in range(2, total_pages + 1):
            spec['_page'] = page_index
            yield self.list_groups(spec, fields=fields, limit=limit)

    def list_groupmembership(self, spec, fields=('person',), one=False, limit=None):
        """
        https://staff-api.yandex-team.ru/v3/groupmembership?_doc=1
        """
        return self._list('groupmembership', spec, fields, one, limit)
