import datetime
import itertools
import logging
import random
import time
from collections import defaultdict


import yp.data_model as data_model

from infra.qyp.account_manager.src.lib.calendar import CalendarClient
from infra.qyp.account_manager.src.lib.startrek import StartrekClient, Ticket
from infra.qyp.account_manager.src import constant
from infra.swatlib.httpclient import HttpClientException
from sepelib.core import config
from yt import yson


log = logging.getLogger(__name__)


class DismissedVmOwnerWorker:

    def __init__(self, zk_storage, yp_client_list):
        self.yp_client_list = yp_client_list
        self.zk_storage = zk_storage
        self.zk_dismissed_owners_by_abc_key = 'dismissed_owners_vms_by_abc'
        self.zk_open_tickets_by_abc_key = 'abc_with_open_ticket'
        self.startrek_cli = StartrekClient(token=config.get_value('startrek.token', None))
        self.calendar_cli = CalendarClient()
        self.startrek_queue = config.get_value('startrek.queue', None)
        self.open_tickets_limit = config.get_value('startrek.open_tickets_limit', 100)

        job_cfg = config.get_value('jobs.dismissed_owners')
        self.blacklist_users = job_cfg.get('blacklist_users', [])
        self.blacklist_accounts = job_cfg.get('blacklist_accounts', [])
        self.blacklist_accounts_for_owning_check = job_cfg.get('blacklist_accounts_for_owning_check', [])
        self.dry_run_tickets = job_cfg.get('dry_run_tickets', True)

    def _push_zookeeper_data(self, json_data, data_id):
        if not self.zk_storage:
            raise NotImplementedError('Not found zookeeper storage instance')
        self.zk_storage.set(json_data, data_id)

    def run(self):
        selectors = ['/meta/id', '/meta/acl', '/meta/account_id']
        query = '[/labels/deploy_engine]=\'QYP\''
        vms_data = dict()

        for yp_ctx_client in self.yp_client_list:
            cluster = yp_ctx_client.cluster
            cluster_name = cluster.get('cluster_name')
            request = yp_ctx_client.select_objects(query, data_model.OT_POD_SET, selectors)
            self._prepare_vm_owners_data(request, vms_data, cluster_name)
        problem_vms = self._check_vm_owners(vms_data)
        self._fill_vm_authors(problem_vms)
        vms_by_abc = self.group_vms_by_abc(problem_vms)
        self._fill_account_admin_members(vms_by_abc)
        self._push_zookeeper_data(vms_by_abc, self.zk_dismissed_owners_by_abc_key)
        self._handle_tickets(vms_by_abc)

    def _prepare_vm_owners_data(self, request, vms_data, cluster_name):
        """
        Parse vm account, users and groups
        :type request: object_service_pb2.TRspSelectObjects
        :type vms_data: dict[str, dict]
        :type cluster_name: None
        :rtype: None
        """
        for item in request.results:
            vm_id, acls, account_id = [yson.loads(val) for val in item.values]
            vm_id = '{}.{}'.format(vm_id, cluster_name)
            vms_data[vm_id] = {
                'account_id': account_id,
                'owner_users': [],
                'owner_groups': []
            }
            owners = acls[1]['subjects'] if len(acls) == 2 else []  # first element in acls is used for robots
            for owner in owners:
                if owner.startswith('abc') or owner.startswith('staff'):
                    vms_data[vm_id]['owner_groups'].append(owner)
                else:
                    vms_data[vm_id]['owner_users'].append(owner)

    def _check_vm_owners(self, vms):
        """
        returns vms with deleted(or without access) logins and groups
        :type vms: dict[str, dict]
        :rtype: dict[str, list]
        """
        res = dict()
        groups_to_check = set()
        users_to_check = set()
        accounts = set()
        blacklist_vms = self.zk_storage.get(constant.dismissed_owner_blacklist_vms) or {}

        for v in vms.values():
            accounts.add(v['account_id'])
            groups_to_check.update(v['owner_groups'])
            users_to_check.update(v['owner_users'])
        client = random.choice(self.yp_client_list)
        # get users allowed to use VM quota(account)
        account_users = client.get_object_access_allowed_for(accounts, data_model.OT_ACCOUNT, data_model.ACA_USE)
        existing_groups = self._get_existing_yp_objects(data_model.OT_GROUP, groups_to_check)
        existing_users = self._get_existing_yp_objects(data_model.OT_USER, users_to_check)

        for vm_id, vm_data in vms.iteritems():
            # skip blacklisted vms
            if vm_id in blacklist_vms:
                continue
            account_id = vm_data['account_id']
            if account_id in self.blacklist_accounts:
                continue

            # check owner groups against deleted groups and owner logins against account users
            for group in vm_data['owner_groups']:
                if group not in existing_groups:
                    if vm_id not in res:
                        res[vm_id] = {
                            'groups': [],
                            'non_account_users': [],
                            'dismissed_users': [],
                            'account_id': account_id
                        }
                    res[vm_id]['groups'].append(group)

            users_info = self._get_dis_users(vm_data, existing_users, account_users)
            if users_info['dismissed_users'] or users_info['non_account_users']:
                if vm_id not in res:
                    res[vm_id] = {
                        'groups': [],
                        'non_account_users': [],
                        'dismissed_users': [],
                        'account_id': account_id
                    }
                res[vm_id]['non_account_users'].extend(users_info['non_account_users'])
                res[vm_id]['dismissed_users'].extend(users_info['dismissed_users'])
        return res

    def _fill_vm_authors(self, problem_vms):
        """
        :type problem_vms: dict
        :rtype: None
        """
        vms_by_cluster = defaultdict(list)
        for vm, data in problem_vms.iteritems():
            if data['account_id'] == constant.QYP_PERSONAL_ID:
                vm_id, cluster = vm.rsplit('.', 1)
                vms_by_cluster[cluster].append(vm_id)
            else:
                data['author'] = None
        for cluster, vm_ids in vms_by_cluster.iteritems():
            self._get_vm_authors(cluster, vm_ids, problem_vms)

    def _get_vm_authors(self, cluster, vm_ids, problem_vms):
        yp_client = [y for y in self.yp_client_list if y.cluster['cluster_name'] == cluster][0]
        for batch in self.split_iterable_into_batches(vm_ids):
            rsp = yp_client.get_objects(batch, data_model.OT_POD, selectors=['/meta/id', '/annotations/owners/author'])
            for r in rsp.subresponses:
                if r.result.values:
                    vm_id, author = yson.loads(r.result.values[0]), yson.loads(r.result.values[1])
                    problem_vms['{}.{}'.format(vm_id, cluster)]['author'] = author if author and author != '#' else None

    @staticmethod
    def group_vms_by_abc(problem_vms):
        """
        :type problem_vms: dict
        :rtype: dict[dict]
        return example:
            {
                'abc:service:1': {
                    'vms': {
                        vm1.SAS: {
                            "account_id": "abc:service:1",
                            "author": "gagl",
                            "non_account_users": ["gagl"],
                            "groups": [],
                            "dismissed_users": ["gagl"]
                        }
                    },
                    'admin_members': [],
                    'name': ''
                }
            }
        """
        d = defaultdict(dict)
        for vm, data in problem_vms.items():
            acc = data['account_id']
            if acc in d:
                d[acc]['vms'][vm] = data
            else:
                d[acc] = {'vms': {vm: data}, 'admin_members': [], 'name': ''}
        return d

    def _handle_tickets(self, vms_by_abc):
        if self.dry_run_tickets:
            log.info('dry running tickets')
            return
        self._close_fixed_tickets(vms_by_abc)
        self._create_new_tickets(vms_by_abc)

    def _close_fixed_tickets(self, vms_by_abc):
        """
        get open tickets and close if fixed. returns vm_ids with open tickets
        :type vms_by_abc: dict
        :rtype: None
        """
        abc_with_open_tickets = self.zk_storage.get(self.zk_open_tickets_by_abc_key) or {}
        open_tickets_count = len(abc_with_open_tickets)
        fixed_abcs = set()
        for abc, data in abc_with_open_tickets.iteritems():
            ticket = Ticket.from_dict(data['ticket'])
            if abc not in vms_by_abc:
                try:
                    self.startrek_cli.close_ticket(ticket.key, 'fixed')
                except HttpClientException as e:
                    if e.resp.status_code == 404:
                        log.warning('ticket: {} deleted or was already closed'.format(ticket.key))
                        fixed_abcs.add(abc)
                    else:
                        log.warning('Not able to close ticket: {}. Will be tried in next iteration'.format(ticket.key))
                else:
                    fixed_abcs.add(abc)
            else:
                new_ticket = self._write_ticket_content(abc, vms_by_abc[abc])
                new_ticket.key = ticket.key
                if self.startrek_cli.compare_and_update_ticket(old_ticket=ticket, new_ticket=new_ticket):
                    log.info('updated ticket for: {}'.format(abc))
                    abc_with_open_tickets[abc]['ticket'] = new_ticket.to_dict()

                if int(time.time() - data['last_warn_time']) > constant.TICKET_IGNORED_REMINDER_TIME:
                    if not self.calendar_cli.is_holiday(date=datetime.datetime.now().strftime('%Y-%m-%d')):
                        self.startrek_cli.comment(ticket, constant.reminder_comment_text)
                        abc_with_open_tickets[abc]['last_warn_time'] = time.time()
                        log.info('send reminder for ticket: {}'.format(ticket.key))
        # delete fixed abcs from storage
        for abc in fixed_abcs:
            del abc_with_open_tickets[abc]
        self._push_zookeeper_data(abc_with_open_tickets, self.zk_open_tickets_by_abc_key)
        log.info('closed {} tickets out of {} open'.format(len(fixed_abcs), open_tickets_count))

    def _create_new_tickets(self, vms_by_abc):
        open_tickets = self.zk_storage.get(self.zk_open_tickets_by_abc_key) or {}
        for abc, abc_data in vms_by_abc.iteritems():
            if len(open_tickets) >= self.open_tickets_limit:
                break
            if abc in open_tickets:
                continue
            ticket = self._write_ticket_content(abc, abc_data)
            try:
                log.info('Creating ticket for: {}'.format(abc))
                created_ticket = self.startrek_cli.create_ticket(ticket)
                open_tickets[abc] = {
                    'ticket': created_ticket.to_dict(), 'creation_time': time.time(), 'last_warn_time': time.time(),
                    'abc_data': abc_data
                }
                # here we push to zk each iteration
                # so if something goes wrong for next vm, already opened tickets will be stored in ZK
                self._push_zookeeper_data(open_tickets, self.zk_open_tickets_by_abc_key)
            except HttpClientException as e:
                log.warning('Failed to create ticket: {}'.format(e.resp.content))

    def _write_ticket_content(self, abc_name, data):
        """
        :type abc_name: str | unicode
        :type data: dict
        :rtype: Ticket
        """
        summary = constant.dismissed_user_ticket_title.format(abc_name=data['name'] if data['name'] else abc_name)
        abc_id = abc_name if abc_name == constant.TEMPORARY_ACCOUNT_ID else abc_name.split(':')[-1]
        desc = constant.dismissed_user_ticket_content_start.format(abc_id=abc_id, abc_name=abc_name)
        for vm, vm_data in sorted(data['vms'].iteritems()):
            vm_id, cluster = vm.rsplit('.', 1)
            desc += constant.dismissed_vm_in_abc_start.format(vm_id=vm_id, cluster=cluster)
            if vm_data['dismissed_users']:
                desc += constant.dissmissed_users_text.format(list=sorted(vm_data['dismissed_users']))
            if vm_data['non_account_users']:
                desc += constant.non_account_users_text.format(list=sorted(vm_data['non_account_users']))
            if vm_data['groups']:
                desc += constant.deleted_groups_text.format(list=sorted(vm_data['groups']))
            if abc_name == constant.QYP_PERSONAL_ID:
                desc += "}>"
            else:
                desc += constant.dissmissed_users_exception_text.format(cluster=cluster, vm_id=vm_id)

        tags = ["account_id:{}".format(abc_id), constant.dismissed_vm_ticket_tag]
        followers = data['admin_members'][:5]
        desc += constant.dismissed_user_ticket_content_end
        assignee = followers[0] if followers else constant.ROOT_FOLLOWER
        res = {
            'queue': self.startrek_queue, 'summary': summary, 'description': desc,
            'tags': tags, 'assignee': assignee, 'followers': followers
        }
        ticket = Ticket.from_dict(res)
        return ticket

    def _get_existing_yp_objects(self, object_type, object_ids):
        """
        :type object_type: int
        :type object_ids: iterable
        :rtype: set
        """
        existing_objects = set()
        client = random.choice(self.yp_client_list)
        for batch in self.split_iterable_into_batches(object_ids):
            rsp = client.get_objects(batch, object_type, selectors=['/meta/id'])
            for r in rsp.subresponses:
                if r.result.values:
                    existing_objects.add(yson.loads(r.result.values[0]))
        return existing_objects

    @staticmethod
    def split_iterable_into_batches(iterable, batch_size=50):
        it = iter(iterable)
        batch = list(itertools.islice(it, batch_size))
        while len(batch) > 0:
            yield batch
            batch = list(itertools.islice(it, batch_size))

    def _get_dis_users(self, vm_data, existing_users, acc_users):
        """
        :type vm_data: dict
        :type existing_users: set
        :type acc_users: dict[str, list]
        :rtype: dict[str, list]
        """
        res = {'dismissed_users': [], 'non_account_users': []}
        acc = vm_data['account_id']
        non_account_users = []
        for user in vm_data['owner_users']:
            if user not in existing_users:
                res['dismissed_users'].append(user)
                continue

            if user in self.blacklist_users or acc in self.blacklist_accounts_for_owning_check:
                continue
            if user not in acc_users[acc]:
                non_account_users.append(user)
        # add non_account_users only if all users don't have access to vm
        if not set(acc_users[acc]).intersection(set(vm_data['owner_users'])):
            res['non_account_users'] = non_account_users

        return res

    def _fill_account_admin_members(self, vms_by_abc):
        """
        :type vms_by_abc: dict
        :rtype: None
        """
        client = random.choice(self.yp_client_list)

        # fill account human readable names
        for batch in self.split_iterable_into_batches(vms_by_abc.iterkeys()):
            rsp = client.get_objects(batch, data_model.OT_GROUP, selectors=['/meta/id', '/meta/name'])
            for r in rsp.subresponses:
                if r.result.values:
                    acc_id, name = yson.loads(r.result.values[0]), yson.loads(r.result.values[1])
                    if name:
                        vms_by_abc[acc_id]['name'] = name

        for scope in constant.ABC_ACCOUNT_SCOPES:
            admin_members = self._get_account_admin_groups(vms_by_abc, admin_group_id=scope)
            for acc, members in admin_members.iteritems():
                vms_by_abc[acc]['admin_members'].extend(members)

    def _get_account_admin_groups(self, vms_by_abc, admin_group_id):
        """
        :type vms_by_abc: dict
        :type admin_group_id: str
        :rtype: dict
        """
        client = random.choice(self.yp_client_list)
        res = {}
        group_ids = []

        for i, d in vms_by_abc.iteritems():
            if i != constant.TEMPORARY_ACCOUNT_ID and i != constant.QYP_PERSONAL_ID and not d['admin_members']:
                group_num = i.rsplit(':', 1)[-1]
                group_admin_id = 'abc:service-scope:{}:{}'.format(group_num, admin_group_id)
                group_ids.append(group_admin_id)

        for batch in self.split_iterable_into_batches(group_ids):
            rsp = client.get_objects(batch, data_model.OT_GROUP, selectors=['/meta/id', '/spec/members'])
            for r in rsp.subresponses:
                if r.result.values:
                    group_admin_id, members = yson.loads(r.result.values[0]), yson.loads(r.result.values[1])
                    if members:
                        acc_num = group_admin_id.rsplit(':', 2)[-2]
                        res['abc:service:{}'.format(acc_num)] = members
        return res
