import logging
from urllib3 import Retry
from requests import Session
from requests.adapters import HTTPAdapter
from cachetools import func


log = logging.getLogger(__name__)


class CalendarClient:
    url = 'https://api.calendar.yandex-team.ru/intapi/get-holidays?from={start_date}&to={end_date}&for=rus'

    @classmethod
    def get_holidays(cls, start_date, end_date):
        """
        dates in %Y-%m-%d format
        :type start_date: str
        :type end_date: str
        :rtype: dict
        """
        retries = Retry(total=5, backoff_factor=0.2)
        s = Session()
        s.mount('https://', adapter=HTTPAdapter(max_retries=retries))
        r = s.get(cls.url.format(start_date=start_date, end_date=end_date))
        r.raise_for_status()
        return r.json()

    @classmethod
    @func.ttl_cache(ttl=60*60*24)
    def is_holiday(cls, date):
        """
        :type date: str
        :rtype: bool
        """
        try:
            holidays = cls.get_holidays(date, date)
            # holidays list is empty
            if not holidays['holidays']:
                return False
            day_type = holidays['holidays'][0]['type']
            if day_type == 'weekend' or day_type == 'holiday':
                return True
            return False
        except Exception as e:
            log.warning('Exception getting holiday for date {}'.format(e))
            # return false if not able to get holidays
            return False
