import time
import uuid
import logging

import requests
from requests.packages.urllib3 import Retry
from requests.adapters import HTTPAdapter


class JNSClient(object):
    MAX_RETRIES = 3

    def __init__(self, api_url, token, project, retry_timeout):
        self.api_url = api_url
        self.jns_project = project
        self.retry_timeout = retry_timeout
        retries = Retry(
            total=self.MAX_RETRIES,
            backoff_factor=1,
            status_forcelist=frozenset((429, 500, 502, 503, 504)),
            method_whitelist=frozenset(('POST',))
        )
        http_adapter = HTTPAdapter(max_retries=retries)
        self.session = requests.Session()
        self.session.mount('http://', adapter=http_adapter)
        self.session.mount('https://', adapter=http_adapter)
        self.session.headers.update({
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(token),
        })

    def send(self, logins, template, params):
        """
        :type logins: list[str]
        :type template: str
        :type params: dict
        :rtype: bool
        """
        req = {
            'project': self.jns_project,
            'template': template,
            'request_id': str(uuid.uuid4()),
            'retry_until': int(time.time()) + self.retry_timeout,
            'params': {key: {'string_value': value} for key, value in params.iteritems()},
            'recipient': {
                'email': {
                    'internal': [{'login': login} for login in logins]
                }
            }
        }
        url = '{}messages/send'.format(self.api_url)
        try:
            with self.session.post(url=url, json=req) as json_rsp:
                json_rsp.raise_for_status()
        except Exception as e:
            logging.warning('JNS returned error: %s%s', e, json_rsp.content)
            return False
        return True
