import logging

from infra.qyp.proto_lib import vmset_pb2

import cachetools
from google.protobuf.json_format import ParseDict
from infra.swatlib.auth import staff
from infra.swatlib.metrics import InstrumentedSession

staff_groups_cache = cachetools.TTLCache(maxsize=300, ttl=30 * 60)
log = logging.getLogger(__name__)


def get_staff_groups(login, staff_client):
    groups = staff_groups_cache.get(login)
    if groups is None:
        try:
            groups = staff.get_group_ids(staff_client, login)
        except Exception as e:
            # Ignore staff errors.
            log.warning('Staff error while fetching user groups: %s', str(e))
            groups = []
        else:
            staff_groups_cache[login] = groups
    return groups


class QDMClient(object):
    DEFAULT_URL = 'https://qdm.yandex-team.ru'
    QDM_TVM_ID = '2015617'
    QDM_STATE_MAP = {
        'new': vmset_pb2.BackupStatus.PLANNED,
        'draft': vmset_pb2.BackupStatus.IN_PROGRESS,
        'active': vmset_pb2.BackupStatus.COMPLETED,
        'archive': vmset_pb2.BackupStatus.REMOVED,
    }
    QDM_ORIGIN_MAP = {
        'evoq': vmset_pb2.BackupSpec.EVOQ,
        'qdm': vmset_pb2.BackupSpec.USER,
        'hot': vmset_pb2.BackupSpec.HOT,
    }

    def __init__(self, tvm_context, url=None):
        self._tvm_context = tvm_context
        self._base_url = url.rstrip('/') if url else self.DEFAULT_URL

    @property
    def session(self):
        session = InstrumentedSession('qdm')
        session.headers['Content-Type'] = 'application/json'
        session.headers['Accept-Encoding'] = ''
        session.headers['X-Ya-Service-Ticket'] = self._tvm_context.ticket_to(dst=self.QDM_TVM_ID)
        return session

    def backup_list(self, vm_id, yp_cluster):
        """
        :type vm_id: str
        :type yp_cluster: str
        :rtype: list[vmset_pb2.Backup]
        """
        resp = self.session.get(self._base_url + '/api/v1/backup_list?vmid={vm_id}&dc={dc}'.format(
            vm_id=vm_id,
            dc=yp_cluster.lower(),
        ))
        resp.raise_for_status()
        result = []
        for item in resp.json():
            backup = vmset_pb2.Backup()
            backup.meta.id = str(item['key'])
            backup.meta.creation_time.FromSeconds(item['create_ts'])
            backup.spec.type = vmset_pb2.BackupSpec.QDM
            backup.spec.cluster = yp_cluster.upper()
            backup.spec.vm_id = vm_id
            backup.spec.generation = item['rev_id']
            backup.status.state = self.QDM_STATE_MAP.get(item['state'])
            backup.status.url = 'qdm:' + item['key']
            result.append(backup)
        return result

    def user_backup_list(self, query, clusters, ctx):
        """
        :type query: vmset_pb2.BackupFindQuery
        :rtype: list[vmset_pb2.Backup]
        """
        params = {
            'user': query.login,
            'groups': get_staff_groups(query.login, ctx),
        }
        resp = self.session.get(self._base_url + '/api/v1/user_backup_list', params=params)
        resp.raise_for_status()
        backups = [self._resp_to_backup(item) for item in resp.json()]
        result = self._filter_user_backup_list(backups, query, clusters)
        return result

    @staticmethod
    def _filter_user_backup_list(backups, query, clusters):
        result = []
        for backup in backups:
            if clusters and backup.spec.cluster not in clusters:
                continue
            backup.status.vm_alive = True
            if query.vm_name:
                if query.vm_name not in backup.spec.vm_id:
                    continue
            if query.segment:
                if backup.spec.vm_spec.qemu.node_segment not in query.segment:
                    continue
            if query.HasField('creation_time_gte'):
                if backup.meta.creation_time.ToSeconds() < query.creation_time_gte.ToSeconds():
                    continue
            if query.HasField('creation_time_lte'):
                if backup.meta.creation_time.ToSeconds() > query.creation_time_lte.ToSeconds():
                    continue
            result.append(backup)
        return result

    def _resp_to_backup(self, item):
        backup = vmset_pb2.Backup()
        vm_id, cluster = item['vm_id'].split('.')
        backup.meta.id = str(item['key'])
        backup.meta.creation_time.FromSeconds(item['create_ts'])
        backup.spec.cluster = cluster.upper()
        backup.spec.vm_id = vm_id
        backup.spec.type = vmset_pb2.BackupSpec.QDM
        backup.spec.generation = item['rev_id']
        backup.spec.origin = self.QDM_ORIGIN_MAP.get(item['origin'])
        backup.status.state = self.QDM_STATE_MAP.get(item['state'])
        backup.status.url = 'qdm:' + item['key']
        vm = vmset_pb2.VM()
        ParseDict(item['vmspec'], vm, ignore_unknown_fields=True)
        backup.spec.vm_meta.CopyFrom(vm.meta)
        backup.spec.vm_spec.CopyFrom(vm.spec)
        if backup.status.state == vmset_pb2.BackupStatus.COMPLETED:
            backup.spec.total_size = sum(file_info['s'] for file_info in item['filemap']['f'])
        return backup
