from __future__ import unicode_literals
import logging

from infra.swatlib.httpclient import HttpClient, HttpClientException


log = logging.getLogger('AccountPoller')


class StartrekClient(object):
    url = 'https://st-api.yandex-team.ru/v2'

    def __init__(self, token):
        self.client = HttpClient(
            client_name='startrek',
            exc_cls=HttpClientException,
            base_url=self.url,
            req_timeout=5,
            token=token,
            max_retries=5
        )

    def create_ticket(self, ticket):
        """
        :type ticket: Ticket
        """
        url = self.url + '/issues/'
        data = {
            'queue': ticket.queue,
            'summary': ticket.summary,
            'description': ticket.description,
        }
        if ticket.get_tags():
            data['tags'] = ticket.get_tags()
        if ticket.assignee:
            data['assignee'] = ticket.assignee
        if ticket.followers:
            data['followers'] = ticket.followers
        res = self.client.post(url, json=data)
        followers = [f['id'] for f in res['followers']] if 'followers' in res else []
        created_ticket = Ticket(
            queue=res['queue']['key'],
            key=res['key'],
            summary=res['summary'],
            description=res['description'],
            assignee=res['assignee']['id'],
            followers=followers,
            tags=res['tags']
        )
        return created_ticket

    def compare_and_update_ticket(self, old_ticket, new_ticket):
        """
        return True if ticket was updated
        :type old_ticket: Ticket
        :type new_ticket: Ticket
        :rtype: bool
        """
        url = self.url + '/issues/{}'.format(old_ticket.key)
        data = dict()
        if old_ticket.description.encode('utf-8') != new_ticket.description:
            data['description'] = new_ticket.description
        if data:
            self.client.put(url, json=data)
            return True
        return False

    def list_tickets(self, queue, unresolved=False):
        """
        :type queue: str
        :type unresolved: bool
        :rtype: collections.Iterable
        """
        url = self.url + '/issues/'
        page = 1
        limit = 100
        params = {
            'filter': ['queue:{}'.format(queue)],
            'perPage': limit
        }
        if unresolved:
            params['filter'].append('resolution:unresolved()')
        while True:
            params['page'] = page
            tickets = self.client.get(url, params=params)
            for ticket in tickets:
                yield ticket
            if len(tickets) != limit:
                break
            page += 1

    def close_ticket(self, ticket_key, resolution):
        data = {'resolution': resolution}
        try:
            url = self.url + '/issues/{}/transitions/closed/_execute'.format(ticket_key)
            return self.client.post(url, json=data)
        except:
            url = self.url + '/issues/{}/transitions/close/_execute'.format(ticket_key)
            return self.client.post(url, json=data)

    def user_will_remove(self, ticket_key):
        data = {}
        url = self.url + '/issues/{}/transitions/onTheSideOfUser/_execute'.format(ticket_key)
        return self.client.post(url, json=data)

    def comment(self, ticket, text):
        """
        :type ticket: Ticket
        :type text: str | unicode
        :rtype: None
        """
        url = self.url + '/issues/{}/comments'.format(ticket.key)
        t = self.get_ticket(ticket.key)
        assignee = t['assignee']['id'] if 'assignee' in t else ticket.assignee
        data = {
            'text': text,
            'summonees': assignee
        }
        self.client.post(url, json=data)

    def get_ticket(self, ticket_key):
        """
        :type ticket_key: str
        :rtype: dict
        """
        url = self.url + '/issues/{}'.format(ticket_key)
        return self.client.get(url)


class Ticket:
    def __init__(self, queue, key, summary, description, assignee, followers, tags):
        self.queue = queue
        self.key = key
        self.summary = summary
        self.description = description
        self.tags = tags
        self.assignee = assignee
        self.followers = followers

    def to_dict(self):
        return {
            'queue': self.queue,
            'key': self.key,
            'summary': self.summary,
            'description': self.description,
            'tags': self.tags,
            'assignee': self.assignee,
            'followers': self.followers
        }

    @classmethod
    def from_dict(cls, data):
        return cls(
            queue=data['queue'],
            key=data.get('key', None),
            summary=data['summary'],
            description=data['description'],
            tags=data['tags'],
            assignee=data['assignee'],
            followers=data['followers']
        )

    def get_tags(self):
        return self.tags if self.tags else []
