import math
from collections import defaultdict

from infra.qyp.account_manager.src import constant
from infra.qyp.proto_lib import vmset_pb2
from infra.qyp.account_manager.src import helpers


class UsageCalculator(object):
    DEV_SEGMENT = 'dev'
    TMP_ACCOUNT = 'tmp'

    def __init__(self, login):
        """
        :type login: str
        """
        self.group_usage_stat = UsageStatistics()
        self.personal_usage = UsageAcc()
        self.total_usage = UsageAcc()
        self.limits = vmset_pb2.ResourceInfo()
        self.login = login
        self.vs_usage_stats = UsageStatistics()

    def init(self, accounts):
        """
        :type accounts: list[vmset_pb2.Account]
        """
        for account in accounts:
            if account.id == self.TMP_ACCOUNT:
                continue
            if account.type == vmset_pb2.Account.PERSONAL:
                if account.id == constant.QYP_PERSONAL_ID:
                    self.limits = account.personal.limits.per_segment[self.DEV_SEGMENT]
                    cluster_usage = account.personal.usage.per_segment[self.DEV_SEGMENT]
                    factor = 1
                    self.personal_usage.plus(cluster_usage, factor)
                    self.total_usage.plus(cluster_usage, factor)
                else:
                    for segment in account.personal.usage.per_segment:
                        cluster_usage = account.personal.usage.per_segment[segment]
                        self.vs_usage_stats.push_service(account.id, cluster_usage, 1, segment)
            elif account.type == vmset_pb2.Account.SERVICE and account.members_count:
                cluster_usage = account.limits.per_segment[self.DEV_SEGMENT]
                factor = float(1) / account.members_count
                self.total_usage.plus(cluster_usage, factor)
                self.group_usage_stat.push_service(account.id, cluster_usage, factor, self.DEV_SEGMENT)
        self.personal_usage.finalize()
        self.total_usage.finalize()
        self.group_usage_stat.finalize()

    def get_available_usage(self):
        """
        :rtype: UsageAcc
        """
        return self.total_usage.diff(self.limits)


class UsageAcc(object):
    def __init__(self, cpu=None, mem=None, hdd=None, ssd=None, addrs=None):
        self.cpu = cpu or 0
        self.mem = mem or 0
        self.disk_per_storage = {'hdd': hdd or 0, 'ssd': ssd or 0}
        self.internet_address = addrs or 0
        self.gpu_per_model = defaultdict(int)
        self.io_guarantees_per_storage = defaultdict(int)

    def plus(self, res, factor):
        """
        :type res: vmset_pb2.ResourceInfo
        :type factor: float
        """
        self.cpu += res.cpu * factor
        self.mem += res.mem * factor
        self.disk_per_storage['hdd'] += res.disk_per_storage['hdd'] * factor
        self.disk_per_storage['ssd'] += res.disk_per_storage['ssd'] * factor
        for model in res.gpu_per_model:
            self.gpu_per_model[model] += res.gpu_per_model[model]
        self.internet_address += res.internet_address * factor
        for storage_type in res.io_guarantees_per_storage:
            self.io_guarantees_per_storage[storage_type] += res.io_guarantees_per_storage[storage_type] * factor

    def diff(self, res):
        """
        :type res: vmset_pb2.ResourceInfo
        :rtype: UsageAcc
        """
        diff = UsageAcc()
        diff.cpu = res.cpu - self.cpu
        diff.mem = res.mem - self.mem
        diff.disk_per_storage = {
            'hdd': res.disk_per_storage['hdd'] - self.disk_per_storage['hdd'],
            'ssd': res.disk_per_storage['ssd'] - self.disk_per_storage['ssd'],
        }
        diff.internet_address = res.internet_address - self.internet_address
        return diff

    def finalize(self):
        self.cpu = self._round(self.cpu, 1000)
        self.mem = self._round(self.mem, 1024 ** 3)
        self.disk_per_storage['hdd'] = self._round(self.disk_per_storage['hdd'], 1024 ** 3)
        self.disk_per_storage['ssd'] = self._round(self.disk_per_storage['ssd'], 1024 ** 3)
        self.internet_address = self._round(self.internet_address)
        for st, val in self.io_guarantees_per_storage.iteritems():
            self.io_guarantees_per_storage[st] = self._round(val, 1024 ** 2)

    def _round(self, value, up_to=1):
        """
        :rtype: int
        """
        return int(math.ceil(value / up_to)) * up_to

    def to_str(self):
        """
        :rtype: str
        """
        return 'CPU: {}, MEM: {}, HDD: {}, SSD: {}, IPv4 {}'.format(
            self.cpu / 1000,
            helpers.int_to_str_gigabytes(self.mem),
            helpers.int_to_str_gigabytes(self.disk_per_storage['hdd']),
            helpers.int_to_str_gigabytes(self.disk_per_storage['ssd']),
            self.internet_address,
        )

    def to_dict(self):
        return {
            'CPU': '{}'.format(self.cpu / 1000),
            'MEM': '{}'.format(helpers.int_to_str_gigabytes(self.mem)),
            'HDD': '{}'.format(helpers.int_to_str_gigabytes(self.disk_per_storage['hdd'])),
            'SSD': '{}'.format(helpers.int_to_str_gigabytes(self.disk_per_storage['ssd'])),
            'IPv4': '{}'.format(self.internet_address)
        }

    def to_dict_byte(self):
        return {
            'CPU': self.cpu,
            'MEM': self.mem,
            'HDD': self.disk_per_storage['hdd'],
            'SSD': self.disk_per_storage['ssd'],
            'IPv4': self.internet_address
        }

    def to_resource_info(self):
        """
        :rtype: vmset_pb2.ResourceInfo
        """
        res = vmset_pb2.ResourceInfo()
        res.cpu = int(self.cpu)
        res.mem = int(self.mem)
        res.disk_per_storage['hdd'] = int(self.disk_per_storage['hdd'])
        res.disk_per_storage['ssd'] = int(self.disk_per_storage['ssd'])
        res.internet_address = int(self.internet_address)
        for model, val in self.gpu_per_model.iteritems():
            res.gpu_per_model[model] = val
        for storage_type in self.io_guarantees_per_storage:
            res.io_guarantees_per_storage[storage_type] = int(self.io_guarantees_per_storage[storage_type])
        return res


class UsageStatistics(object):
    def __init__(self):
        self.usage_accounts = dict()  # {service_id: {segment: UsageAcc}}
        self.summary = UsageAcc()

    def push_service(self, service_id, limits, factor, segment):
        """
        :type service_id: str
        :type limits: vmset_pb2.ResourceInfo
        :type factor: float
        :type segment: str
        """
        cpu = limits.cpu * factor
        mem = limits.mem * factor
        hdd = limits.disk_per_storage['hdd'] * factor
        ssd = limits.disk_per_storage['ssd'] * factor
        addrs = limits.internet_address * factor
        if cpu + mem + hdd + ssd + addrs != 0:
            if service_id not in self.usage_accounts:
                self.usage_accounts[service_id] = {segment: UsageAcc()}
            elif segment not in self.usage_accounts[service_id]:
                self.usage_accounts[service_id][segment] = UsageAcc()

            self.usage_accounts[service_id][segment].plus(limits, factor)

    def finalize(self):
        for segment_usage in self.usage_accounts.itervalues():
            for usage in segment_usage.itervalues():
                self.summary.plus(usage, 1)
        self.summary.finalize()

    def get_statistics(self):
        return self.usage_accounts

    def get_group_usage(self):
        return self.summary
