import logging
import os
import re

import flask
import gevent
import yp.client
from flask import Flask
from infra.qyp.account_manager.src import account_poller, constant
from infra.qyp.account_manager.src.contrib import abc_client, staff_client
from infra.qyp.account_manager.src.lib import qdm_client
from infra.qyp.account_manager.src.lib import jns_client
from infra.qyp.account_manager.src.lib.zk_storage import ZkStorage
from infra.qyp.account_manager.src.model import pod_controller, yp_client
from infra.qyp.account_manager.src.web.account_service import am_service_bp
from infra.qyp.account_manager.src.web.am_views import am_bp
from infra.qyp.vmctl.src.api import VMProxyClient
from infra.swatlib import rpc
from infra.swatlib import webserver
from infra.swatlib.auth import oauth
from infra.swatlib.auth import passport
from infra.swatlib.auth import tvm
from infra.swatlib.gevent import exclusiveservice
from infra.swatlib.zk import client as zk_client
from sepelib.core import config
from sepelib.util.net import mail as mail_client

log = logging.getLogger('app')
VERSION = 'development version'
DEFAULT_TRANSPORT = 'http'


class Ctx(object):
    """
    Web application context passed via flask.g attribute.
    """
    def __init__(self, rpc_authenticator, abc_client, mail_client, staff_client, vmproxy_client, yp_client_list,
                 pod_ctl_map, zk_storage, tvm_client, qdm_client, jns_client):
        self.rpc_authenticator = rpc_authenticator
        self.abc_client = abc_client
        self.mail_client = mail_client
        self.staff_client = staff_client
        self.vmproxy_client = vmproxy_client
        self.yp_client_list = yp_client_list
        self.pod_ctl_map = pod_ctl_map
        self.zk_storage = zk_storage
        self.tvm_client = tvm_client
        self.qdm_client = qdm_client
        self.personal_quota = config.get_value('personal_limit', None)
        self.jns_client = jns_client


class Application(object):
    name = 'account-manager'

    def init_http_server(self, ctx):
        app = Flask(__name__)
        app.register_blueprint(am_service_bp)
        app.register_blueprint(am_bp)

        @app.before_request
        def add_ctx():
            flask.g.ctx = ctx

        allowed_origins = config.get_value('web.http.cors.allowed_origins', [])
        allowed_origins_regex = config.get_value('web.http.cors.allowed_origins_regex', [])

        @app.after_request
        def add_diagnostics_headers(response):
            response.headers['Access-Control-Allow-Methods'] = 'PUT, DELETE'
            response.headers['Access-Control-Expose-Headers'] = 'etag,x-total-items'
            response.headers['Access-Control-Allow-Headers'] = 'content-type,if-match'
            response.headers['Access-Control-Allow-Credentials'] = 'true'
            origin = flask.request.environ.get('HTTP_ORIGIN')
            if origin in allowed_origins:
                response.headers['Access-Control-Allow-Origin'] = origin
            elif origin and any(re.match(domain_pattern, origin) for domain_pattern in allowed_origins_regex):
                response.headers['Access-Control-Allow-Origin'] = origin
            return response

        return webserver.WebServer(config.get(), app, version=VERSION)

    def make_coord(self, instance_id):
        return zk_client.ZookeeperClient(
            cfg={
                'hosts': config.get_value('coord.hosts'),
                'zk_root': config.get_value('coord.root'),
                'read_only': False,
                'log_debug': config.get_value('coord.log_debug'),
            },
            identifier=instance_id,
        )

    def _make_ctx(self):
        yp_token = config.get_value('yp.token', None)
        default_transport = config.get_value('yp.default_transport', DEFAULT_TRANSPORT)
        pod_ctl_map = {}
        yp_client_list = []
        for yp_cfg in config.get_value('yp.address'):
            cluster = yp_cfg.get('cluster_name')
            address = yp_cfg.get('url')
            transport = yp_cfg.get('transport', default_transport)
            if 'token' not in yp_cfg:
                yp_cfg['token'] = yp_token
            client_base = yp.client.YpClient(address=address, config=yp_cfg, transport=transport)
            stub = client_base.create_grpc_object_stub()
            yp_ctl = yp_client.YpClient(stub=stub, cluster=yp_cfg)
            pod_ctl_map[cluster] = pod_controller.PodController(address, yp_ctl)
            yp_client_list.append(yp_ctl)

        tvm_client = tvm.TvmClient(
            client_id=config.get_value('tvm.client_id'),
            secret=os.environ.get('TVM_SECRET') or config.get_value('tvm.secret'),
            api_url=config.get_value('tvm.api_url')
        )
        oauth_client = oauth.OAuth.from_config(config.get_value('oauth'))
        passport_client = passport.TvmPassportClient.from_config(
            d=config.get_value('passport'),
            tvm_client=tvm_client
        )
        rpc_authenticator = rpc.authentication.CachingPassportAuthenticator(
            oauth_client=oauth_client,
            passport_client=passport_client,
            is_auth_disabled=not config.get_value('run.auth'),
            force_return_user=config.get_value('run.force_return_user')
        )
        jns = jns_client.JNSClient(
            api_url=config.get_value('jns.api_url'),
            token=config.get_value('jns.token'),
            project=config.get_value('jns.project'),
            retry_timeout=config.get_value('jns.retry_timeout'),
        )

        return Ctx(
            rpc_authenticator=rpc_authenticator,
            abc_client=abc_client.AbcClient(oauth_token=config.get_value('abc.token', None)),
            mail_client=mail_client.Mail.from_config(config.get_value('mail', constant.DEFAULT_EMAIL_CONFIG)),
            staff_client=staff_client.StaffClient(oauth_token=config.get_value('staff.token')),
            vmproxy_client=VMProxyClient(token=yp_token),
            yp_client_list=yp_client_list,
            pod_ctl_map=pod_ctl_map,
            zk_storage=self.zk_storage,
            tvm_client=tvm_client,
            qdm_client=qdm_client.QDMClient(tvm_client),
            jns_client=jns,
        )

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.coord = self.make_coord(self.instance_id)
        self.zk_storage = ZkStorage(config.get_value('coord.root'), self.coord)
        ctx = self._make_ctx()
        plr = account_poller.AccountPoller(ctx)
        plr.set_zk_storage(self.zk_storage)
        self.plr = exclusiveservice.ExclusiveService(self.coord, self.name, plr)
        self.web = self.init_http_server(ctx)

    def run(self):
        logging.getLogger('urllib3.connectionpool').setLevel(logging.ERROR)
        logging.getLogger("yt.packages.urllib3.connectionpool").setLevel(logging.ERROR)

        gevent.spawn(self.web.run)
        self.coord.start().wait()
        self.plr.start()
        self.plr.wait()

    def stop(self):
        self.plr.stop()
        self.web.stop()


if __name__ == '__main__':
    path_to_config = os.path.abspath(__file__).split('/')[:-2]
    path_to_config.append('cfg_default.yml')
    config.load('/'.join(path_to_config))
    app = Application(0000)
    app.run()
