import cachetools.func

from collections import defaultdict
from yt import yson
from infra.qyp.proto_lib import vmset_pb2
from infra.qyp.account_manager.src import helpers
from infra.qyp.account_manager.src import constant
from infra.qyp.account_manager.src.model import abc_roles


class FakeContext:
    def __init__(self, abc_client, pod_ctl, personal_quotas_dict):
        self.abc_client = abc_client
        self.personal_quotas_dict = personal_quotas_dict
        self.pod_ctl = pod_ctl


def set_personal_limits(limits, personal_quota):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type personal_quota: dict
    """
    per_segment_limits = limits.per_segment[personal_quota['segment']]
    per_segment_limits.mem = personal_quota['mem']
    per_segment_limits.cpu = personal_quota['cpu']
    for disk in personal_quota['disk']:
        per_segment_limits.disk_per_storage[disk['storage']] = disk['capacity']
        per_segment_limits.io_guarantees_per_storage[disk['storage']] = disk['bandwidth_guarantee']
    per_segment_limits.internet_address = personal_quota['internet_address']


def set_personal_usage(usage, vms, login):
    """
    :type usage: vmset_pb2.ResourceTotals
    :type vms: list[vmset_pb2.Vm]
    :type login: str
    """
    for vm in vms:
        if vm.meta.author != login:
            continue
        per_segment_usage = usage.per_segment[vm.spec.qemu.node_segment]
        per_segment_usage.cpu += vm.spec.qemu.resource_requests.vcpu_guarantee
        per_segment_usage.mem += vm.spec.qemu.resource_requests.memory_guarantee
        for volume in vm.spec.qemu.volumes:
            per_segment_usage.disk_per_storage[volume.storage_class] += volume.capacity
        if vm.spec.qemu.enable_internet:
            per_segment_usage.internet_address += 1
        for storage_class, value in vm.spec.qemu.io_guarantees_per_storage.items():
            per_segment_usage.io_guarantees_per_storage[storage_class] += value


def set_personal_usage_new(usage, segment, dict_usage):
    per_segment_usage = usage.per_segment[segment]
    per_segment_usage.cpu = dict_usage['CPU']
    per_segment_usage.mem = dict_usage['MEM']
    if dict_usage['HDD'] != 0:
        per_segment_usage.disk_per_storage['hdd'] = dict_usage['HDD']
    if dict_usage['SSD'] != 0:
        per_segment_usage.disk_per_storage['ssd'] = dict_usage['SSD']
    per_segment_usage.internet_address = dict_usage['IPv4']


def cast_yp_resources(pb, yp_resources):
    """
    :type pb: vmset_pb2.ResourceInfo
    :type yp_resources: data_model_pb2.TPerSegmentResourceTotals
    """
    pb.mem = yp_resources.memory.capacity
    pb.cpu = yp_resources.cpu.capacity
    for key, value in yp_resources.disk_per_storage_class.iteritems():
        pb.disk_per_storage[key] = value.capacity
        pb.io_guarantees_per_storage[key] = value.bandwidth
    pb.internet_address = yp_resources.internet_address.capacity
    for key, value in yp_resources.gpu_per_model.iteritems():
        pb.gpu_per_model[key] = value.capacity


def set_limits(limits, account, segment=None):
    """
    :type limits: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.spec.resource_limits.per_segment:
        cast_yp_resources(limits.per_segment[segment], account.spec.resource_limits.per_segment[segment])
    else:
        for segment_, segment_limits in account.spec.resource_limits.per_segment.iteritems():
            cast_yp_resources(limits.per_segment[segment_], segment_limits)


def set_usage(usage, account, segment=None):
    """
    :type usage: vmset_pb2.ResourceTotals
    :type account: data_model_pb2.TAccount
    :type segment: str | NoneType
    """
    if segment and segment in account.status.resource_usage.per_segment:
        cast_yp_resources(usage.per_segment[segment], account.status.resource_usage.per_segment[segment])
    else:
        for segment_, segment_usage in account.status.resource_usage.per_segment.iteritems():
            cast_yp_resources(usage.per_segment[segment_], segment_usage)


@cachetools.func.ttl_cache(maxsize=1, ttl=3 * 60 * 60)
def get_search_members(pod_ctl):
    """
    :rtype: frozenset
    """
    rsp = pod_ctl.yp_client.get_group(constant.QYP_PERSONAL_ID, selectors=['/spec/members'])
    return frozenset(yson.loads(rsp.result.values[0]))


@cachetools.func.ttl_cache(maxsize=1, ttl=3 * 60 * 60)
def get_list_users_abc(abc_client, accounts, use_cache):
    account_ids = accounts.split('#')
    result_list_users = abc_roles.get_service_qyp_persons(abc_client, account_ids, use_cache)
    return result_list_users


@cachetools.func.ttl_cache(maxsize=1, ttl=5 * 60)
def get_yp_superusers(ctx):
    """
    :rtype: frozenset
    """
    rsp = ctx.pod_ctl.yp_client.get_group('superusers', selectors=['/spec/members'])
    return frozenset(yson.loads(rsp.result.values[0]))


def match_pods_with_users(meta_pods, users):
    """
    Match pods and usets and return calculated resources
    :param meta_pods: TRspSelectObjects
    :param users: list[str]
    :return: dict[str, dict]
    """
    pods_users_data = defaultdict(dict)
    for user in users:
        pods_users_data[user]['summary'] = {
            'CPU': 0,
            'MEM': 0,
            'HDD': 0,
            'SSD': 0,
            'IPv4': 0
        }

    for pod in meta_pods.results:
        author = yson.loads(pod.values[2])
        if bool(author) and author in users:
            pod_id = yson.loads(pod.values[0])
            spec = yson.loads(pod.values[1])

            cpu = spec['resource_requests']['vcpu_guarantee']
            mem = spec['resource_requests']['memory_guarantee']
            hdd = (spec['disk_volume_requests'][1]['quota_policy']['capacity']
                   if spec['disk_volume_requests'][1]['storage_class'] == 'hdd' else 0)
            ssd = (spec['disk_volume_requests'][1]['quota_policy']['capacity']
                   if spec['disk_volume_requests'][1]['storage_class'] == 'ssd' else 0)
            ip = 0
            for ipv6 in spec['ip6_address_requests']:
                if ipv6.get('enable_internet') and ipv6['enable_internet']:
                    ip += 1

            # FIXME: remove addition 6 * 1024 ** 3 after correct display resources used
            infra_gap = 6 * 1024 ** 3
            if spec['disk_volume_requests'][1]['storage_class'] == 'hdd':
                pods_users_data[author][pod_id] = helpers.make_dict_resources_bytes(cpu, mem, hdd + infra_gap, ssd, ip)
                pods_users_data[author]['summary']['HDD'] += hdd + infra_gap
            else:
                pods_users_data[author][pod_id] = helpers.make_dict_resources_bytes(cpu, mem, hdd, ssd + infra_gap, ip)
                pods_users_data[author]['summary']['SSD'] += ssd + infra_gap
            pods_users_data[author]['summary']['CPU'] += cpu
            pods_users_data[author]['summary']['MEM'] += mem
            pods_users_data[author]['summary']['IPv4'] += ip

    return pods_users_data


def run(ctx, logins=None, segment=None, use_cache=None):
    """
    :type ctx: vmproxy.web.app.Ctx or FakeContext
    :type logins: list[str]
    :type segment: str
    :type use_cache: bool
    :rtype: list[vmset_pb2.Account]
    """
    selectors = ['/meta/id',
                 '/spec',
                 '/annotations/owners/author']
    meta_pods = ctx.pod_ctl.list_pods('abc:service:4172', logins, selectors)
    pods_users_data = match_pods_with_users(meta_pods, logins)

    logins_su = set(logins).intersection(get_yp_superusers(ctx))
    if logins_su:
        service_ids = abc_roles.list_user_services_from_abc(ctx.abc_client, logins_su)
        account_ids = {'abc:service:{}'.format(service_id) for service_id in service_ids}
        user_account_dict = ctx.pod_ctl.list_account_ids_by_logins(set(logins) - logins_su)
        for login in logins_su:
            user_account_dict[login] = account_ids
    else:
        user_account_dict = ctx.pod_ctl.list_account_ids_by_logins(logins)

    for user, user_account_ids in user_account_dict.items():
        user_account_dict[user] = abc_roles.filter_accounts_by_roles(
            abc_client=ctx.abc_client,
            account_ids=user_account_ids,
            login=user,
            use_cache=use_cache
        )

    account_ids = set().union(*user_account_dict.values())
    accounts = ctx.pod_ctl.list_accounts(account_ids)

    selectors = ['/meta/id',
                 '/spec/members']
    groups_rsp = ctx.pod_ctl.list_groups(account_ids, selectors)
    members_dict = {yson.loads(item.values[0]): item.values[1] for item in groups_rsp.results}
    result = defaultdict(list)
    cache = defaultdict(set)
    result_list_users = get_list_users_abc(ctx.abc_client, '#'.join(account_ids), use_cache)
    search_members = get_search_members(pod_ctl=ctx.pod_ctl)

    for user, user_account_ids in user_account_dict.items():
        for account in accounts:
            if account.meta.id not in user_account_ids:
                continue
            service_id = account.meta.id.split('abc:service:')[-1]
            acc = vmset_pb2.Account()
            acc.id = account.meta.id
            set_limits(acc.limits, account, segment)
            set_usage(acc.usage, account, segment)
            if acc.id in members_dict:
                if cache.get(acc.id) is None:
                    cache[acc.id] = result_list_users[service_id] or list(yson.loads(members_dict[acc.id]))
                acc.members_count = sum(int(member in search_members) for member in cache[acc.id])
            if account.meta.id in ctx.personal_quotas_dict:
                acc.type = vmset_pb2.Account.PERSONAL
                set_personal_limits(acc.personal.limits, ctx.personal_quotas_dict[account.meta.id])
                set_personal_usage_new(acc.personal.usage, segment, pods_users_data[user]['summary'])
            else:
                acc.type = vmset_pb2.Account.SERVICE
            result[user].append(acc)
    return [pods_users_data, result]
