import yt_yson_bindings
import yp.data_model as data_model

from infra.qyp.account_manager.src.lib.gutils import idle_iter
from infra.qyp.account_manager.src.model import abc_roles
from infra.qyp.account_manager.src.model import yputil
from infra.qyp.account_manager.src import helpers
from yt import yson


def yp_error_dict_to_str(error):
    """
    :type error: dict
    :rtype: str
    """
    message = ""
    if "message" in error:
        message = error["message"]

    for inner_error in error.get("inner_errors", []):
        message += ': {}'.format(yp_error_dict_to_str(inner_error))
    return message


TYPE_MODEL_MAP = {
    data_model.OT_POD: data_model.TPod,
    data_model.OT_POD_SET: data_model.TPodSet,
}


def loads_proto(data, proto_class):
    return yt_yson_bindings.loads_proto(data, proto_class=proto_class, skip_unknown_fields=True)


class PodController(object):
    def __init__(self, yp_cluster, yp_client):
        """
        :type yp_cluster: str
        :type yp_client: YpClient
        """
        self.yp_cluster = yp_cluster
        self.yp_client = yp_client

    def get_object(self, object_id, object_type, timestamp=None):
        """
        :type object_id: str
        :type object_type: int
        :type timestamp: int | NoneType
        :rtype: data_model.TPod
        """
        rsp = self.yp_client.get_object(object_id, object_type, timestamp)
        return loads_proto(rsp.result.values[0], TYPE_MODEL_MAP[object_type])

    def _cast_yson_to_pod(self, obj):
        pod = data_model.TPod(
            spec=loads_proto(obj[0], data_model.TPodSpec),
        )
        annotations = {
            'owners': yson.loads(obj[1]),
        }
        yputil.cast_dict_to_attr_dict(annotations, pod.annotations)
        return pod

    def _cast_pod_yson_to_owners_dict(self, obj):
        return yson.loads(obj[4])

    def _cast_yson_to_pod_set(self, obj):
        pod_set = data_model.TPodSet(
            meta=loads_proto(obj[0], data_model.TPodSetMeta),
            spec=loads_proto(obj[1], data_model.TPodSetSpec)
        )
        yputil.cast_dict_to_attr_dict(yson.loads(obj[2]), pod_set.labels)
        return pod_set

    def _get_pod_set_items(self, query, timestamp=None, skip=0, limit=0):
        """
        :type query: str
        :type timestamp: int
        :rtype: list[data_model.TPodSet]
        """
        selectors = [
            '/meta',
            '/spec',
            '/labels',
        ]
        rsp = self.yp_client.list_pod_sets(query=query, timestamp=timestamp, selectors=selectors, offset=skip,
                                           limit=limit)

        return [self._cast_yson_to_pod_set(item.values) for item in idle_iter(rsp.results)]

    def check_use_account_permission(self, acc_id, subject_id, use_cache, abc_client):
        """
        :type acc_id: str
        :type subject_id: str
        :type use_cache: bool
        :type abc_client: AbcClient
        :rtype: bool
        """
        if acc_id == 'tmp':
            return True
        has_permission = self.yp_client.check_object_permissions(
            object_id=acc_id,
            object_type=data_model.OT_ACCOUNT,
            subject_id=subject_id,
            permission=[data_model.ACA_USE]
        )
        if not has_permission:
            return False
        return acc_id in abc_roles.filter_accounts_by_roles(abc_client, [acc_id], subject_id, use_cache)

    def list_accounts(self, account_ids):
        """
        :type query: str
        :rtype: list[data_model.TAccount]
        """
        if len(account_ids):
            query = '[/meta/id] in ({})'.format(','.join(helpers.quoted(account_ids)))
        else:
            query = '[/meta/id] in ("")'

        rsp = self.yp_client.list_accounts(query)
        accounts = []
        for result in rsp.results:
            accounts.append(loads_proto(result.values[0], data_model.TAccount))
        return accounts

    def get_accounts(self, ids, selectors=None):
        """
        :type ids: list[str]
        :type selectors: list[str]
        :rtype: list[data_model.TAccount]
        """
        rsp = self.yp_client.get_objects(ids, data_model.OT_ACCOUNT)
        accounts = []
        for result in rsp.subresponses:
            if result.result.values and len(result.result.values) > 0:
                accounts.append(yputil.loads_proto(result.result.values[0], data_model.TAccount))
        return accounts

    def list_account_ids_by_logins(self, logins):
        """
        :type logins: list[str]
        :rtype: dict[str, list[str]]
        """
        return self.yp_client.get_user_access_allowed_to(logins, data_model.OT_ACCOUNT, data_model.ACA_USE)

    def list_all_pod_sets_for_logins(self, logins):
        """
        :param logins: list[str]
        :return: list[str]
        """
        pod_sets_dict = self.yp_client.get_user_access_allowed_to(
            users=logins,
            object_type=data_model.OT_POD_SET,
            permission=data_model.ACA_GET_QYP_VM_STATUS
        )
        return list(set().union(*pod_sets_dict.values()))

    def list_pod_sets_for_account(self, account_id):
        """
        :param account_id: str
        :return: List[str]
        """
        q = '[/meta/account_id] = "{}"'.format(account_id)
        rsp = self.yp_client.select_objects(q, data_model.OT_POD_SET, selectors=['/meta/id'])
        res = [yson.loads(v.values[0]) for v in rsp.results]
        return res

    def list_groups(self, account_ids, selectors=None):
        """
        :param account_ids: list[str]
        :param selectors: list[str] or None
        :return: TRspSelectObjects
        """
        if len(account_ids) > 0:
            query = '[/meta/id] in ({})'.format(','.join(helpers.quoted(account_ids)))
        else:
            query = '[/meta/id] in ("")'

        return self.yp_client.list_groups(query, selectors=selectors)

    def get_pods(self, pod_ids, selectors=None):
        """
        :param pod_ids: list[str]
        :param selectors: list[str] or None
        :return: TRspSelectObjects
        """
        if len(pod_ids) > 0:
            query = '[/meta/id] in ({})'.format(','.join(helpers.quoted(pod_ids)))
        else:
            query = '[/meta/id] in ("")'

        return self.yp_client.list_pods(query, selectors=selectors)

    def list_pods(self, account_id, logins, selectors=None, segment=None):
        """
        :type account_id: str
        :type logins: list[str]
        :type selectors: list[str] or None
        :type segment: list[str] or None
        :return: TRspSelectObjects
        """
        user_access_pod_set_ids = self.list_all_pod_sets_for_logins(logins)
        if user_access_pod_set_ids:
            pod_set_items = self._get_pod_set_items(
                query='[/labels/deploy_engine] = "QYP" AND [/spec/account_id] = "{}" AND [/meta/id] in ({})'.format(
                    account_id, ','.join('"{}"'.format(x) for x in user_access_pod_set_ids)
                )
            )
        else:
            pod_set_items = []
        pod_set_ids = [ps.meta.id for ps in pod_set_items]
        return self.get_pods(pod_set_ids, selectors)

    def check_object_permissions_few(self, object_id, subject_ids):
        """
        :param object_id: str
        :param subject_ids: list[str]
        :return: list[str, data_model.ACA_DENY or data_model.ACA_ALLOW]
        """
        return self.yp_client.check_object_permissions_few(object_id, data_model.OT_ACCOUNT, subject_ids,
                                                           data_model.ACA_USE)

    def get_existing_pods(self, pod_ids):
        """
        :type pod_ids: list
        :rtype: set[str]
        """
        res = set()
        rsp = self.yp_client.get_objects(pod_ids, data_model.OT_POD, selectors=['/meta/id'], ignore_nonexistent=True)
        for r in rsp.subresponses:
            if len(r.result.values) > 0:
                res.add(yson.loads(r.result.values[0]))
        return res
