from yp_proto.yp.client.api.proto import object_service_pb2
import yp.data_model as data_model
from infra.swatlib.gevent import geventutil


class IYpClient(object):
    """
    Interface to be used in dependency injection.
    """


class YpClient(IYpClient):
    def __init__(self, stub, cluster):
        self.stub = stub
        self.cluster = cluster

    def select_objects(self, query, object_type, selectors, timestamp=None, offset=0, limit=0):
        """
        :type query: str | NoneType
        :type object_type: int
        :type selectors: list[str]
        :type timestamp: int | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        if limit:
            req.limit.value = limit
        if offset:
            req.offset.value = offset
        if query:
            req.filter.query = query
        if timestamp:
            req.timestamp = timestamp
        req.selector.paths.extend(selectors)
        return self.stub.SelectObjects(req)

    def check_object_permissions_few(self, object_id, object_type, subjects_id, permission):
        """
        :type object_id: str
        :type object_type: int
        :type subjects_id: list[str]
        :type permission: int
        :rtype: list[[str, data_model.ACA_ALLOW or data_model.ACA_DENY]]
        """
        req = object_service_pb2.TReqCheckObjectPermissions()
        for subject_id in geventutil.gevent_idle_iter(subjects_id):
            sub = req.subrequests.add()
            sub.object_id = object_id
            sub.object_type = object_type
            sub.subject_id = subject_id
            sub.permission = permission
        rsp = self.stub.CheckObjectPermissions(req)
        return zip(subjects_id, [i.action for i in rsp.subresponses])

    def check_object_permissions(self, object_id, object_type, subject_id, permission):
        """
        :type object_id: str
        :type object_type: int
        :type subject_id: str
        :type permission: list[int]
        :rtype: bool
        """
        req = object_service_pb2.TReqCheckObjectPermissions()
        for p in permission:
            sub = req.subrequests.add()
            sub.object_id = object_id
            sub.object_type = object_type
            sub.subject_id = subject_id
            sub.permission = p
        rsp = self.stub.CheckObjectPermissions(req)

        for subresponse in rsp.subresponses:
            if subresponse.action == data_model.ACA_ALLOW:
                return True

        return False

    def update_object(self, object_id, object_type, set_updates, transaction_id=None):
        """
        :type object_id: str
        :type object_type: int
        :type set_updates:  dict[str, str]
        :type transaction_id: str | NoneType
        """
        req = object_service_pb2.TReqUpdateObject()
        if transaction_id is not None:
            req.transaction_id = transaction_id
        req.object_id = object_id
        req.object_type = object_type
        for path, value in set_updates.iteritems():
            update = req.set_updates.add()
            update.path = path
            update.value = value
        self.stub.UpdateObject(req)

    def update_objects(self, object_ids, object_type, set_updates, transaction_id=None):
        """
        :type object_ids: list[str]
        :type object_type: int
        :type set_updates:  list[dict[str, str]]
        :type transaction_id: str | NoneType
        """
        req = object_service_pb2.TReqUpdateObjects()
        if transaction_id is not None:
            req.transaction_id = transaction_id
        for object_id, set_update in zip(object_ids, set_updates):
            sub = req.subrequests.add()
            sub.object_type = object_type
            sub.object_id = object_id
            for path, value in set_update.iteritems():
                update = sub.set_updates.add()
                update.path = path
                update.value = value
        self.stub.UpdateObjects(req)

    def update_pod(self, object_id, set_updates, transaction_id=None):
        """
        :type transaction_id: str
        :type object_id: str
        :type set_updates:  dict[str, str]
        """
        self.update_object(object_id, data_model.OT_POD, set_updates, transaction_id)

    def update_pods(self, object_ids, set_updates, transaction_id=None):
        """
        :type transaction_id: str
        :type object_ids: list[str]
        :type set_updates:  list[dict[str, str]]
        """
        self.update_objects(object_ids, data_model.OT_POD, set_updates, transaction_id)

    def get_user_access_allowed_to(self, users, object_type, permission):
        """
        :type users: list[str]
        :type object_type: int
        :type permission: int
        :rtype: dict[str, list[str]]
        """
        req = object_service_pb2.TReqGetUserAccessAllowedTo()
        for user in users:
            sub = req.subrequests.add()
            sub.user_id = user
            sub.object_type = object_type
            sub.permission = permission
        rsp = self.stub.GetUserAccessAllowedTo(req)
        return {user: list(s.object_ids) for user, s in zip(users, rsp.subresponses)}

    def list_accounts(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_ACCOUNT, selectors, timestamp)

    def list_pods(self, query, timestamp=None, selectors=None, offset=0, limit=0):
        """
        :type query: str
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_POD, selectors, timestamp, offset=offset, limit=limit)

    def list_groups(self, query, timestamp=None, selectors=None):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_GROUP, selectors, timestamp)

    def list_pod_sets(self, query, timestamp=None, selectors=None, offset=0, limit=0):
        """
        :type query: str | NoneType
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :type limit: int
        :type offset: int
        :rtype: object_service_pb2.TRspSelectObjects
        """
        selectors = selectors or ['']
        return self.select_objects(query, data_model.OT_POD_SET, selectors, timestamp, offset=offset, limit=limit)

    def get_object(self, object_id, object_type, timestamp=None, selectors=None):
        """
        :type object_id: str
        :type object_type: int
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: data_model.TRspGetObject
        """
        req = object_service_pb2.TReqGetObject()
        req.object_type = object_type
        req.object_id = object_id
        if not selectors:
            req.selector.paths.append('')
        else:
            req.selector.paths.extend(selectors)
        if timestamp:
            req.timestamp = timestamp
        return self.stub.GetObject(req)

    def get_node_segment(self, segment, selectors=None):
        """
        :type segment: str
        :type selectors: list[str] | None
        :rtype data_model.TRspGetObject
        """
        selectors = selectors or ['']
        return self.get_object(segment, data_model.OT_NODE_SEGMENT, selectors=selectors)

    def get_group(self, object_id, timestamp=None, selectors=None):
        """
        :type object_id: str
        :type timestamp: int | NoneType
        :type selectors: list[str] | NoneType
        :rtype: object_service_pb2.TRspGetObject
        """
        return self.get_object(object_id, data_model.OT_GROUP, timestamp, selectors)

    def get_objects(self, object_ids, object_type, timestamp=None, selectors=None, ignore_nonexistent=True):
        """
        :type object_ids: list[str]
        :type object_type: int
        :type timestamp: int | NoneType
        :type selectors: list[str]
        :type ignore_nonexistent: bool
        :rtype: object_service_pb2.TRspGetObjects
        """
        req = object_service_pb2.TReqGetObjects()
        req.object_type = object_type
        if selectors:
            req.selector.paths.extend(selectors)
        else:
            req.selector.paths.append('')
        if timestamp:
            req.timestamp = timestamp
        for tmp in object_ids:
            sub = req.subrequests.add()
            sub.object_id = tmp
        req.options.ignore_nonexistent = ignore_nonexistent
        return self.stub.GetObjects(req)

    def generate_timestamp(self):
        """
        :rtype: int
        """
        req = object_service_pb2.TReqGenerateTimestamp()
        rsp = self.stub.GenerateTimestamp(req)
        return rsp.timestamp

    def get_object_access_allowed_for(self, object_ids, object_type, permission):
        """
        :type object_ids: list[str]
        :type object_type: int
        :type permission: int
        :rtype: dict[str, list[str]]
        """

        req = object_service_pb2.TReqGetObjectAccessAllowedFor()
        for object_id in object_ids:
            sub = req.subrequests.add()
            sub.object_id = object_id
            sub.object_type = object_type
            sub.permission = permission
        rsp = self.stub.GetObjectAccessAllowedFor(req)
        return {obj: set(s.user_ids) for obj, s in zip(object_ids, rsp.subresponses)}
