from collections import defaultdict
import logging
from infra.qyp.proto_lib import accounts_api_pb2, accounts_pb2, vmset_pb2

import cachetools
import flask
from yt.packages.requests import exceptions as yt_request_exceptions
from infra.qyp.account_manager.src import constant
from infra.qyp.account_manager.src.action import list_user_accounts as list_user_accounts_action
from infra.qyp.account_manager.src.lib import usage_calculator
from infra.qyp.account_manager.src.lib.rpc import blueprint
from infra.qyp.account_manager.src.model.actions.get_account_summary import get_account_summary
from infra.swatlib.rpc import exceptions


am_service_bp = blueprint.HttpRpcBlueprint('rpc.account_service', __name__, url_prefix='/')
log = logging.getLogger(__name__)
access_response_cache = cachetools.TTLCache(maxsize=100, ttl=3 * 60 * 60)


def get_context():
    return flask.g.ctx


@am_service_bp.method('GetAccountQuotaSummary',
                      request_type=accounts_api_pb2.GetAccountQuotaSummaryRequest,
                      response_type=accounts_api_pb2.GetAccountQuotaSummaryResponse,
                      need_authentication=False)
def get_account_quota_summary(req, _):
    if req.account_id and req.logins:
        raise exceptions.BadRequestError('Only one of two parameters must be selected (logins or account_id)')

    key = (req.account_id, frozenset(req.exclude_logins), frozenset(req.exclude_scopes), frozenset(req.logins))
    r = access_response_cache.get(key)
    if r is not None:
        return r

    result = get_account_summary(
        ctx=get_context(),
        service_id=req.account_id,
        exclude_users=req.exclude_logins,
        exclude_scopes=req.exclude_scopes,
        list_logins=req.logins
    )
    r = accounts_api_pb2.GetAccountQuotaSummaryResponse(account_quota_summary=result)
    access_response_cache[key] = r
    return r


@am_service_bp.method('ListQuotaRequest',
                      request_type=accounts_api_pb2.ListQuotaRequestRequest,
                      response_type=accounts_api_pb2.ListQuotaRequestResponse)
def list_quota_request(req, _):
    resp = accounts_api_pb2.ListQuotaRequestResponse()

    r = resp.results.add()
    r.meta.id = '1'
    r.meta.author = 'frolstas'
    r.meta.creation_time.GetCurrentTime()
    r.meta.last_modification_time.GetCurrentTime()
    r.spec.source_id = ''
    r.spec.recipient_id = ''
    r.spec.resources.cpu = str(10 * 10 ** 3)
    r.spec.resources.memory = str(3 * 1024 ** 3)
    r.spec.resources.hdd = str(0)
    r.spec.resources.ssd = str(100 * 1024 ** 3)
    r.spec.resources.ip = str(0)
    r.status.state = accounts_pb2.CONFIRMED

    r = resp.results.add()
    r.meta.id = '2'
    r.meta.author = 'frolstas'
    r.meta.creation_time.GetCurrentTime()
    r.meta.last_modification_time.GetCurrentTime()
    r.spec.source_id = ''
    r.spec.recipient_id = ''
    r.spec.resources.cpu = str(10 * 10 ** 3)
    r.spec.resources.memory = str(3 * 1024 ** 3)
    r.spec.resources.hdd = str(0)
    r.spec.resources.ssd = str(100 * 1024 ** 3)
    r.spec.resources.ip = str(0)
    r.status.state = accounts_pb2.REJECTED

    r = resp.results.add()
    r.meta.id = '3'
    r.meta.author = 'frolstas'
    r.meta.creation_time.GetCurrentTime()
    r.meta.last_modification_time.GetCurrentTime()
    r.spec.source_id = ''
    r.spec.recipient_id = ''
    r.spec.resources.cpu = str(10 * 10 ** 3)
    r.spec.resources.memory = str(3 * 1024 ** 3)
    r.spec.resources.hdd = str(0)
    r.spec.resources.ssd = str(100 * 1024 ** 3)
    r.spec.resources.ip = str(0)
    r.status.state = accounts_pb2.NEW
    return resp


@am_service_bp.method('CreateQuotaRequest',
                      request_type=accounts_api_pb2.CreateQuotaRequestRequest,
                      response_type=accounts_api_pb2.CreateQuotaRequestResponse)
def create_quota_request(req, auth):
    resp = accounts_api_pb2.CreateQuotaRequestResponse()
    resp.request_id = str(1)
    return resp


@am_service_bp.method('GetQuotaRequest',
                      request_type=accounts_api_pb2.GetQuotaRequestRequest,
                      response_type=accounts_api_pb2.GetQuotaRequestResponse)
def get_quota_request(req, _):
    if not req.request_id:
        raise exceptions.BadRequestError('No request_id specified')
    resp = accounts_api_pb2.GetQuotaRequestResponse()
    r = resp.result
    r.meta.id = '1'
    r.meta.author = 'frolstas'
    r.meta.creation_time.GetCurrentTime()
    r.meta.last_modification_time.GetCurrentTime()
    r.spec.source_id = ''
    r.spec.recipient_id = ''
    r.spec.resources.cpu = str(10 * 10 ** 3)
    r.spec.resources.memory = str(3 * 1024 ** 3)
    r.spec.resources.hdd = str(0)
    r.spec.resources.ssd = str(100 * 1024 ** 3)
    r.spec.resources.ip = str(0)
    r.status.state = accounts_pb2.NEW
    return resp


@am_service_bp.method('UpdateQuotaRequest',
                      request_type=accounts_api_pb2.UpdateQuotaRequestRequest,
                      response_type=accounts_api_pb2.UpdateQuotaRequestResponse)
def update_quota_request(req, auth):
    if not req.meta.id:
        raise exceptions.BadRequestError('No meta.id specified')
    resp = accounts_api_pb2.UpdateQuotaRequestResponse()
    return resp


@am_service_bp.method('RemoveQuotaRequest',
                      request_type=accounts_api_pb2.RemoveQuotaRequestRequest,
                      response_type=accounts_api_pb2.RemoveQuotaRequestResponse)
def remove_quota_request(req, auth):
    if not req.request_id:
        raise exceptions.BadRequestError('No request_id specified')
    resp = accounts_api_pb2.RemoveQuotaRequestResponse()
    return resp


@am_service_bp.method('ListBackup',
                      request_type=accounts_api_pb2.ListBackupRequest,
                      response_type=accounts_api_pb2.ListBackupResponse)
def list_backup(req, auth_subject):
    ctx = get_context()
    if not req.vm_id:
        raise exceptions.BadRequestError('No vm_id specified')
    try:
        pod_ctl = ctx.pod_ctl_map[req.cluster]
    except KeyError:
        raise exceptions.BadRequestError('Cluster name "{}" is not valid'.format(req.cluster))

    skip = req.skip or 0
    limit = req.limit or 1000

    backups = ctx.qdm_client.backup_list(req.vm_id, req.cluster)
    try:
        existing_pods = pod_ctl.get_existing_pods([req.vm_id])
        for b in backups:
            if b.spec.vm_id in existing_pods:
                b.status.vm_alive_status = vmset_pb2.BackupStatus.AliveStatus.ALIVE
            else:
                b.status.vm_alive_status = vmset_pb2.BackupStatus.AliveStatus.REMOVE
    except yt_request_exceptions.RequestException as e:
        log.warning('Exception getting pods from %s, vm_alive_statuses will be marked UNKNOWN: %s', req.cluster, e)
    return accounts_api_pb2.ListBackupResponse(backups=backups[skip:skip + limit])


@am_service_bp.method('ListUserBackups',
                      request_type=accounts_api_pb2.ListUserBackupsRequest,
                      response_type=accounts_api_pb2.ListUserBackupsResponse)
def list_user_backups(req, auth_subject):
    if not req.query.login:
        raise exceptions.BadRequestError('No query.login specified')
    skip = req.skip or 0
    limit = req.limit or 1000

    ctx = get_context()
    backup_list = ctx.qdm_client.user_backup_list(req.query, req.clusters, ctx.staff_client)
    backups_by_cluster = defaultdict(list)
    for b in backup_list:
        backups_by_cluster[b.spec.cluster].append(b)
    for cluster in backups_by_cluster:
        pod_ctl = ctx.pod_ctl_map.get(cluster)
        if pod_ctl is None:
            continue
        backups = backups_by_cluster[cluster]
        try:
            ids = list({b.spec.vm_id for b in backups})
            existing_pods = pod_ctl.get_existing_pods(ids)
            for b in backups:
                if b.spec.vm_id in existing_pods:
                    b.status.vm_alive_status = vmset_pb2.BackupStatus.AliveStatus.ALIVE
                else:
                    b.status.vm_alive_status = vmset_pb2.BackupStatus.AliveStatus.REMOVE
        except yt_request_exceptions.RequestException as e:
            log.warning('Exception getting pods from %s, vm_alive_statuses will be marked UNKNOWN: %s', req.clusters, e)

    result = []
    if not req.query.vm_alive:
        result = backup_list
    else:
        alive_only = req.query.vm_alive == vmset_pb2.BackupFindQuery.ALIVE_ONLY
        removed_only = req.query.vm_alive == vmset_pb2.BackupFindQuery.REMOVED_ONLY
        for b in backup_list:
            vm_alive = b.status.vm_alive_status == vmset_pb2.BackupStatus.AliveStatus.ALIVE
            vm_removed = b.status.vm_alive_status == vmset_pb2.BackupStatus.AliveStatus.REMOVE
            if alive_only and vm_alive or removed_only and vm_removed:
                result.append(b)

    return accounts_api_pb2.ListUserBackupsResponse(backups=result[skip:skip + limit])


@am_service_bp.method('ListUserAccounts',
                      request_type=accounts_api_pb2.ListUserAccountsRequest,
                      response_type=accounts_api_pb2.ListUserAccountsResponse)
def list_user_accounts(req, _):
    if not req.login:
        raise exceptions.BadRequestError('No login specified')
    if req.login == 'root':
        raise exceptions.BadRequestError('Invalid login: {}'.format(req.login))
    ctx = get_context()
    rsp = accounts_api_pb2.ListUserAccountsResponse()
    accounts = []

    accounts_by_cluster = list_user_accounts_action.run_for_clusters(req, ctx)
    for cluster, acs in accounts_by_cluster.iteritems():
        result = accounts_api_pb2.AccountsByCluster()
        result.cluster = cluster
        result.accounts.extend(acs)
        rsp.accounts_by_cluster.extend([result])
        accounts.extend(acs)

    pa_helper = usage_calculator.UsageCalculator(login=req.login)
    pa_helper.init(accounts)
    if constant.QYP_PERSONAL_ID in ctx.personal_quota:
        personal = rsp.personal_summary.add()
        personal.account_id = constant.QYP_PERSONAL_ID
        for i, a in pa_helper.group_usage_stat.usage_accounts.items():
            personal.group_usage_stat[i].CopyFrom(a['dev'].to_resource_info())
        personal.group_usage_stat['summary'].CopyFrom(pa_helper.group_usage_stat.get_group_usage().to_resource_info())
        personal.total_usage.per_segment['dev'].CopyFrom(pa_helper.total_usage.to_resource_info())
    for vs_account, usage_per_segment in pa_helper.vs_usage_stats.get_statistics().iteritems():
        vs_summary = rsp.personal_summary.add()
        vs_summary.account_id = vs_account
        for segment in usage_per_segment:
            vs_summary.total_usage.per_segment[segment].CopyFrom(usage_per_segment[segment].to_resource_info())
    return rsp
