import json
import os

import flask
from flask import Response, request
from infra.qyp.account_manager.src import constant
from infra.swatlib.gevent import exclusiveservice
from infra.swatlib.rpc import request_limiter
from infra.swatlib.rpc.blueprint import make_authentication_request
from sepelib.core import config
from sepelib.flask.auth.util import login_exempt
import yp.data_model as data_model
from yt.orm.library.common import NoSuchObjectError


am_bp = flask.Blueprint('api.account_manager', __name__)


@am_bp.before_request
def auth_request():
    authenticator = flask.g.ctx.rpc_authenticator
    if flask.request.endpoint:
        view = flask.current_app.view_functions.get(flask.request.endpoint)
        if not getattr(view, 'need_auth', True):
            auth_subject = authenticator.ANON_RESULT
        else:
            _auth_request = make_authentication_request(flask.request)
            auth_subject = authenticator.authenticate_request(_auth_request)
    else:
        auth_subject = authenticator.ANON_RESULT

    flask.g.identity = auth_subject


@am_bp.route('/ping')
def ping():
    return 'pong\n'


@am_bp.route('/vm_outside_sp')
@request_limiter.limit(30)
def vm_outside_sp():
    ctx = flask.g.ctx
    return Response(json.dumps(ctx.zk_storage.get('vm_outside_sp'), indent=2), status=200,
                    content_type='application/json')


@am_bp.route('/dismissed_vm')
@request_limiter.limit(30)
def dismissed_vm():
    ctx = flask.g.ctx
    return Response(json.dumps(ctx.zk_storage.get('dismissed_vm'), indent=2), status=200,
                    content_type='application/json')


@am_bp.route('/overquoting_accounts')
@request_limiter.limit(30)
def overquoting_accounts():
    ctx = flask.g.ctx
    return Response(json.dumps(ctx.zk_storage.get('overquoting_accounts'), indent=2), status=200,
                    content_type='application/json')


@am_bp.route('/unistat')
@request_limiter.limit(30)
@login_exempt
def unistat():
    ctx = flask.g.ctx
    res_usage = ctx.zk_storage.get('resource_usage_metrics')
    unused_vms = ctx.zk_storage.get('table_unused_vms_all')
    unused_vms_buckets = {}
    for cluster, pods in unused_vms.iteritems():
        for pod_id, pod_value in pods.iteritems():
            value = pod_value['unused_days']
            if value not in unused_vms_buckets:
                unused_vms_buckets[value] = 0
            unused_vms_buckets[value] += 1
    signals = []
    signals.extend(res_usage)
    signals.extend([['unused_vms_ahhh', unused_vms_buckets.items()]])
    return Response(json.dumps(signals, indent=2), status=200, content_type='application/json')


@am_bp.route('/get_usage_statistics')
@request_limiter.limit(30)
def get_usage_statistics():
    ctx = flask.g.ctx
    result = ctx.zk_storage.get('resource_usage_statistics')
    cluster = request.args.get('cluster', '')
    resource = request.args.get('resource', '')
    segment = request.args.get('segment', 'dev')
    requested = result.get(segment.lower(), {}).get(cluster.upper(), {}).get(resource.lower(), [])
    res = {
        'meta': {
            'segment': segment,
            'count': len(requested)
        },
        'result': requested
    }
    return Response(json.dumps(res, indent=2), status=200, content_type='application/json')


@am_bp.route('/get_unused_vms')
@request_limiter.limit(30)
def get_unused_vms():
    ctx = flask.g.ctx
    result = ctx.zk_storage.get('table_unused_vms_all')
    limit_days = int(request.args.get('days', constant.DEFAULT_DAYS))
    res = {}
    for cluster in result.iterkeys():
        res[cluster] = []
        for pod_id in result[cluster].iterkeys():
            if result[cluster][pod_id]['unused_days'] > limit_days:
                res[cluster].append({
                    'pod_id': pod_id,
                    'unused_days': result[cluster][pod_id]['unused_days']
                })
    return Response(json.dumps(res, indent=2), status=200, content_type='application/json')


@am_bp.route('/get_unused_vms_raw')
@request_limiter.limit(30)
def get_unused_vms_raw():
    login = flask.g.identity.login
    if login not in config.get_value('root_users'):
        return Response('Forbidden', status=403)
    ctx = flask.g.ctx
    result = ctx.zk_storage.get('table_unused_vms_all')
    return Response(json.dumps(result, indent=2), status=200, content_type='application/json')


@am_bp.route('/get_unused_vms_stages')
@request_limiter.limit(30)
def get_unused_vms_stages():
    login = flask.g.identity.login
    if login not in config.get_value('root_users'):
        return Response('Forbidden', status=403)
    ctx = flask.g.ctx
    result = ctx.zk_storage.get('table_unused_vms')
    return Response(json.dumps(result, indent=2), status=200, content_type='application/json')


@am_bp.route('/list_instances')
@request_limiter.limit(30)
def list_instances():
    ctx = flask.g.ctx
    path = os.path.join(exclusiveservice.ExclusiveService.ZK_PATH, 'account-manager')
    lock = ctx.zk_storage.zookeeper.lock(path)
    result = lock.contenders()
    return Response(json.dumps(result), status=200, content_type='application/json')


@am_bp.route('/get_dismissed_owner_blacklist_vms')
@request_limiter.limit(30)
def get_dismissed_owner_blacklist_vms():
    ctx = flask.g.ctx
    result = ctx.zk_storage.get(constant.dismissed_owner_blacklist_vms) or dict()
    result = {'result': result}
    return Response(json.dumps(result), status=200, content_type='application/json')


@am_bp.route('/add_dismissed_owner_blacklist_vm', methods=['POST'])
@request_limiter.limit(30)
def add_dismissed_owner_blacklist_vm():
    ctx = flask.g.ctx
    login = flask.g.identity.login
    cluster, vm_id = flask.request.form.get('cluster', '').upper(), flask.request.form.get('vm_id', '')
    if not cluster or not vm_id:
        result = {'message': 'Missing one of the required fields "cluster" or "vm_id"'}
        return Response(json.dumps(result), status=400, content_type='application/json')
    if cluster not in ctx.pod_ctl_map:
        result = {'message': 'Wrong cluster name "{}"'.format(cluster)}
        return Response(json.dumps(result), status=400, content_type='application/json')
    pod_ctl = ctx.pod_ctl_map[cluster]
    try:
        pod_set = pod_ctl.get_object(object_id=vm_id, object_type=data_model.OT_POD_SET)
    except NoSuchObjectError:
        result = {'message': '"{}" not found in cluster "{}"'.format(vm_id, cluster)}
        return Response(json.dumps(result), status=404, content_type='application/json')
    if pod_set.meta.account_id == constant.QYP_PERSONAL_ID:
        result = {'message': 'not allowed for VMs in personal account'}
        return Response(json.dumps(result), status=403, content_type='application/json')
    is_owner = pod_ctl.yp_client.check_object_permissions(object_id=vm_id,
                                                          object_type=data_model.OT_POD,
                                                          subject_id=login,
                                                          permission=[data_model.ACA_GET_QYP_VM_STATUS])
    if not is_owner and login not in config.get_value('root_users'):
        result = {'message': 'Permission denied. User "{}" does not have acl for VM "{}"'.format(login, vm_id)}
        return Response(json.dumps(result), status=403, content_type='application/json')
    blacklist = ctx.zk_storage.get(constant.dismissed_owner_blacklist_vms) or dict()
    key = '{}.{}'.format(vm_id, cluster)
    if key in blacklist:
        result = {'message': 'VM "{}" was already added to blacklist'.format(vm_id)}
        return Response(json.dumps(result), status=400, content_type='application/json')
    blacklist[key] = login
    ctx.zk_storage.set(blacklist, constant.dismissed_owner_blacklist_vms)
    result = {'message': 'OK'}
    return Response(json.dumps(result), status=200, content_type='application/json')


@am_bp.route('/add_vm_exception_form')
def add_vm_exception_form():
    form_content = constant.dismissed_owner_blacklist_form.format(vm_id=flask.request.args.get('vm_id', ''),
                                                                  cluster=flask.request.args.get('cluster', ''))
    return Response(form_content, status=200)
