import click
import time
from fabric import Connection as SSHConnection
from collections import defaultdict
from infra.qyp.deploy import utils, sandbox_ctl
from infra.qyp.vmctl.src import api as vmproxy_api
from infra.qyp.proto_lib import vmset_api_pb2, vmset_pb2, vmagent_pb2
import nanny_rpc_client


class DeployObject(object):
    sandbox_ctl = None  # type: sandbox_ctl.SandboxCtl


pass_deploy_object = click.make_pass_decorator(DeployObject, ensure=True)


@click.group('qyp-deploy', invoke_without_command=True)
@click.option('--sandbox-token', '-T', type=click.STRING, envvar='SANDBOX_TOKEN')
@click.option('--sandbox-user', '-U', type=click.STRING, envvar='SANDBOX_USER')
@pass_deploy_object
def main(obj, sandbox_token, sandbox_user):
    obj.sandbox_ctl = sandbox_ctl.SandboxCtl(
        token=sandbox_token,
        user=sandbox_user,
    )


@main.command('build-tasks-resource-archive')
@pass_deploy_object
def build_tasks_resource_archive(deploy_obj):  # type: (DeployObject, ...) -> None
    qyp_tasks_directory = 'sandbox/projects/qyp'
    success, patch_info = utils.create_patch_for_sandbox_task(qyp_tasks_directory)
    if not success:
        if patch_info.uncommitted_changes:
            click.echo('Directory {} has uncommitted changes:'.format(qyp_tasks_directory), err=True)
            for _change in patch_info.uncommitted_changes:
                click.echo('\t{status}\t{path}'.format(**_change), err=True)
        raise click.Abort()

    click.echo("Create patch: {}".format(patch_info))
    start_at = time.time()
    task_id = deploy_obj.sandbox_ctl.build_sandbox_tasks(
        svn_revision=patch_info.trunk_svn_revision,
        apply_patch=patch_info.paste_url
    )

    click.echo('Created task https://sandbox.yandex-team.ru/task/{}'.format(task_id))

    for task_status in deploy_obj.sandbox_ctl.wait_task(task_id, idle_period=10):
        minutes = int((time.time() - start_at) / 60)
        seconds = int(time.time() - start_at) % 60
        click.echo('\rExec. time: {}m:{}s, current status: {}'.format(minutes, seconds, task_status), nl=False)
    click.echo('')

    task_status = deploy_obj.sandbox_ctl.get_task_status(task_id)

    if task_status != 'SUCCESS':
        logs_url = deploy_obj.sandbox_ctl.get_task_logs_url(task_id, 'common.log')
        click.echo('BUILD_SANDBOX_TASKS: {}, logs: {}'.format(task_status, logs_url), err=True)
        raise click.Abort()

    result_id = deploy_obj.sandbox_ctl.get_task_resource_id(task_id, 'SANDBOX_TASKS_ARCHIVE')
    click.echo('BUILD_SANDBOX_TASKS Success: Resource SANDBOX_TASKS_ARCHIVE: {}'.format(result_id))


@main.command('build-vmagent')
@click.option('--version', type=click.STRING, required=True)
@click.option('--apply-patch', type=click.STRING, default=None)
@click.option('--tasks-archive-resource', type=click.STRING, default=None)
@pass_deploy_object
def build_vmagent(deploy_obj, version, apply_patch, tasks_archive_resource):  # type: (DeployObject, any, ...) -> None
    patch_info = utils.PatchInfo(trunk_svn_revision=None, paste_url=None, uncommitted_changes=None)
    if apply_patch:
        success, patch_info = utils.create_patch_for_sandbox_task(apply_patch)
        if not success:
            if patch_info.uncommitted_changes:
                click.echo('Directory {} has uncommitted changes:'.format(apply_patch), err=True)
                for _change in patch_info.uncommitted_changes:
                    click.echo('\t{status}\t{path}'.format(**_change), err=True)
            raise click.Abort()
        else:
            click.echo("Create paste url: {}".format(patch_info))

    start_at = time.time()
    task_id = deploy_obj.sandbox_ctl.create_task(
        task_type='BUILD_VMAGENT_V2',
        task_owner='QYP',
        tasks_archive_resource=tasks_archive_resource,
        description='Build VMAgent: {}'.format(version),
        arcadia_rev=patch_info.trunk_svn_revision,
        arcadia_patch=patch_info.paste_url,
        version=version,
        use_aapi_fuse=True,
    )

    click.echo('Created task https://sandbox.yandex-team.ru/task/{}'.format(task_id))

    for task_status in deploy_obj.sandbox_ctl.wait_task(task_id, idle_period=20):
        minutes = int((time.time() - start_at) / 60)
        seconds = int(time.time() - start_at) % 60
        click.echo('\rExec. time: {}m:{}s, current status: {}'.format(minutes, seconds, task_status), nl=False)
    click.echo('')
    task_status = deploy_obj.sandbox_ctl.get_task_status(task_id)

    if task_status != 'SUCCESS':
        logs_url = deploy_obj.sandbox_ctl.get_task_logs_url(task_id, 'common.log')
        click.echo('BUILD_VMAGENT_V2: {}, logs: {}'.format(task_status, logs_url), err=True)
        raise click.Abort()

    vmagent_pack_skynet_id = deploy_obj.sandbox_ctl.get_task_resource_skynet_id(task_id, 'VMAGENT_PACK_V2')
    click.echo('BUILD_VMAGENT_V2: SUCCESS, resource VMAGENT_PACK_V2: {}'.format(vmagent_pack_skynet_id))


@main.command('update-vmagents')
@click.option('-T', '--token', required=True)
@click.option('--vmagent-version-filter', type=click.STRING)
@click.option('--announcement-filter', type=click.STRING)
@click.option('--limit-per-cluster', type=click.INT, default=5)
@click.option('--clusters', type=click.STRING, default='SAS,MAN,VLA', help='Enter clusters, coma separated')
@click.option('--run-update-vmagent', is_flag=True, default=False)
@click.option('--mark-announcement', type=click.STRING, default=None)
@click.option('--group-by-logins', type=click.STRING, default=None)
@click.option('--group-by-node', type=click.STRING, default=None)
def update_vmagents(token, vmagent_version_filter, announcement_filter, clusters, run_update_vmagent,
                    limit_per_cluster, mark_announcement, group_by_logins, group_by_node):
    vmproxy_client = vmproxy_api.VMProxyClient(token=token)
    labels_filters = {}
    if vmagent_version_filter:
        labels_filters['vmagent_version'] = vmset_pb2.YpVmFindQueryBuilder(equal=vmagent_version_filter)
    if announcement_filter:
        labels_filters['qyp_vm_mark/announcement'] = vmset_pb2.YpVmFindQueryBuilder(equal=announcement_filter)
    i = 0

    login_to_vms = defaultdict(list)
    node_to_vms = defaultdict(list)
    for cluster in clusters.split(','):

        vms = vmproxy_client.list_yp_vms(cluster=cluster, labels_filters=labels_filters, limit=limit_per_cluster)
        for vm in vms:
            # if vm.meta.id.startswith('edadeal-'):
            #     continue
            i += 1
            try:
                vm_status_info = vmproxy_client.get_status(cluster=cluster, pod_id=vm.meta.id)
                pod_state_active = True
                vm_state_str = vmagent_pb2.VMState.VMStateType.Name(vm_status_info.state.type)
                vmagent_version = vm_status_info.vmagent_version or vm.spec.vmagent_version
            except nanny_rpc_client.exceptions.BadRequestError as e:
                _msg_starts_with = 'Agent has not been started yet. Current status: '
                vm_state_str = e.message.replace(_msg_starts_with, '')
                pod_state_active = False
                vmagent_version = vm.spec.vmagent_version
            except Exception as e:
                vm_state_str = e.message
                pod_state_active = False
                vmagent_version = vm.spec.vmagent_version

            action = 'skip' if pod_state_active else 'not valid'
            if run_update_vmagent and pod_state_active:
                resp = vmproxy_client.update_vmagent(cluster=cluster, pod_id=vm.meta.id)
                action = " update vmagent {} -> {}".format(
                    vm.spec.vmagent_version,
                    resp.vm.spec.vmagent_version
                )
            for login in vm.meta.auth.owners.logins:
                login_to_vms[login].append(vm.meta.id)

            node_to_vms[vm.status.scheduling.node_id].append(vm.meta.id)
            if 'announcement' in vm.spec.labels:
                action = '{}'.format(vm.spec.labels['announcement'])

            if mark_announcement and pod_state_active:
                vmproxy_client.update_labels(cluster=cluster, pod_id=vm.meta.id, labels={
                    'announcement': mark_announcement
                })
                action = "mark for announcement: {}".format(mark_announcement)

            click.echo('{i}\t{cluster}\t{vm.meta.id:35}\t{vmagent_version}\t{vm_state_str:30}\t{action}'.format(
                vm=vm,
                i=i,
                cluster=cluster,
                pod_state_active=pod_state_active,
                vmagent_version=vmagent_version,
                vm_state_str=vm_state_str,
                action=action
            ))

    if group_by_logins:
        for i, (node, vms) in enumerate(sorted(group_by_logins.items()), start=1):
            click.echo("{}\t{}\t{}".format(i, node, ",".join(vms)))

    if group_by_node:
        for i, (node, vms) in enumerate(sorted(node_to_vms.items()), start=1):
            click.echo("{}\t{}\t{}".format(i, node, ",".join(vms)))


@main.command('sync-vmagents-updates')
@click.argument('token', required=True)
@click.option('--clusters', type=click.STRING, default='SAS,MAN,VLA', help='Enter clusters, coma separated')
@click.option('--announcement-filter', type=click.STRING)
@click.option('--check-ssh', is_flag=True, default=False)
def sync_vmagents_updates(token, clusters, announcement_filter, check_ssh):
    vmproxy_client = vmproxy_api.VMProxyClient(token=token)
    labels_filters = {
        'qyp_vm_mark/update_vmagent_process': vmset_pb2.YpVmFindQueryBuilder(is_not_null=True)
    }
    if announcement_filter:
        labels_filters['qyp_vm_mark/announcement'] = vmset_pb2.YpVmFindQueryBuilder(equal=announcement_filter)
    global_i = 0

    for cluster in clusters.split(','):
        vms = vmproxy_client.list_yp_vms(cluster=cluster, labels_filters=labels_filters, limit=100)
        for i, vm in enumerate(vms):
            update_vmagent_process = vm.spec.labels['update_vmagent_process']
            try:
                vm_status_info = vmproxy_client.get_status(cluster=cluster, pod_id=vm.meta.id)
                autorun = vm_status_info.config.autorun
                vm_type = vm_status_info.config.type
                vm_state_str = vmagent_pb2.VMState.VMStateType.Name(vm_status_info.state.type)
                vmagent_version = vm_status_info.vmagent_version
            except nanny_rpc_client.exceptions.BadRequestError as e:
                _msg_starts_with = 'Agent has not been started yet. Current status: '
                vm_state_str = e.message.replace(_msg_starts_with, '')
                vmagent_version = 'in_progress'
                autorun = None
                vm_type = None
            except Exception as e:
                vm_state_str = e.message
                vmagent_version = 'failed'
                autorun = None
                vm_type = None

            if vm.spec.vmagent_version == vmagent_version and update_vmagent_process != 'done':
                vmproxy_client.update_labels(cluster, pod_id=vm.meta.id, labels={
                    'update_vmagent_process': 'done'
                })
                update_vmagent_process = '-> done'

            check_ssh_result = None
            if check_ssh:
                vm_ip = None
                for ip_allocation in vm.status.ip_allocations:
                    if ip_allocation.owner == 'vm' and ip_allocation.vlan_id == 'backbone':
                        vm_ip = ip_allocation.address
                        break
                if not vm_ip:
                    check_ssh_result = 'ip not found'
                else:
                    try:
                        connection = SSHConnection(vm_ip, connect_timeout=3)
                        resp = connection.run("echo 1", hide=True)
                        check_ssh_result = resp.stdout.strip() == '1'
                    except Exception as e:
                        check_ssh_result = e.message

            if check_ssh and check_ssh_result is True:
                continue
            if check_ssh and vm_type == 1:
                continue

            global_i += 1
            click.echo('{global_i} \t {cluster}:{i}\t{link:75}\t{vmagent_version:>15}'
                       '\t{vm_state_str:>30}\t{update_vmagent_process}'
                       '\tautorun:{autorun}'
                       '\tvm_type:{vm_type}'
                       '\tssh:{check_ssh_result}'.format(
                vm=vm,
                i=i,
                link="https://qyp.yandex-team.ru/vm/{cluster_lower}/{vm.meta.id}?tab=vmstats".format(
                    vm=vm,
                    cluster_lower=cluster.lower(),
                ),
                cluster=cluster,
                vmagent_version=vmagent_version,
                vm_state_str=vm_state_str,
                update_vmagent_process=update_vmagent_process,
                autorun=autorun,
                check_ssh_result=check_ssh_result,
                vm_type=vm_type,
                global_i=global_i
            ))


if __name__ == '__main__':
    main(obj=DeployObject())
