#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time
import logging
import requests
import sandbox.common.types.task as ctt

from sandbox.common.rest import Client as SandboxClient
from sandbox.common.proxy import OAuth

SANDBOX_TIMEOUT = 60 * 20
DOWNLOAD_SPEED = 40


class SandboxCtl(object):

    def __init__(self, token, user, url=None, timeout=SANDBOX_TIMEOUT):
        self._sandbox = SandboxClient(base_url=url, auth=OAuth(token), total_wait=timeout)
        self._sandbox_user = user

    def build_sandbox_tasks(self, svn_revision, apply_patch, task_owner=None):  # type: (str, str, str) ->  int
        custom_fields = {
            'tasks_svn_url': 'arcadia:/arc/trunk/arcadia/sandbox@{}'.format(svn_revision),
            'arcadia_patch': apply_patch
        }
        task_data = self._sandbox.task({'type': 'BUILD_SANDBOX_TASKS'})
        task_id = task_data['id']
        task_input_params = {
            'owner': task_owner or self._sandbox_user,
            'custom_fields': [{'name': k, 'value': v} for k, v in custom_fields.items()],
            'description': 'Apply patch {}'.format(apply_patch),
            'priority': ('USER', 'NORMAL'),
        }
        self._sandbox.task[task_id] = task_input_params
        logging.info('Created task https://sandbox.yandex-team.ru/task/{}'.format(task_id))

        resp = self._sandbox.batch.tasks.start.update(task_id)[0]
        if resp['status'] not in ('SUCCESS', 'WARNING'):
            raise Exception('Failed to create task: {}'.format(resp))

        return task_id

    def create_task(self, task_type, description, task_owner=None, tasks_archive_resource=None, arcadia_rev=None,
                    arcadia_patch=None, **parameters):
        logging.info('Creating Sandbox task...')
        custom_fields = [
            {
                'name': 'checkout_arcadia_from_url',
                'value': 'arcadia:/arc/trunk/arcadia{}'.format(
                    '' if not arcadia_rev else "@{}".format(arcadia_rev)),
            }
        ]
        if arcadia_patch:
            custom_fields.append({
                'name': 'arcadia_patch',
                'value': arcadia_patch
            })

        for name, value in parameters.items():
            custom_fields.append({'name': name, 'value': value})

        task_data = self._sandbox.task({'type': task_type})
        task_id = task_data['id']
        task_input_params = {
            'owner': task_owner or self._sandbox_user,
            'custom_fields': custom_fields,
            'description': description,
            'priority': ('USER', 'NORMAL'),
        }
        if tasks_archive_resource:
            task_input_params['tasks_archive_resource'] = tasks_archive_resource

        self._sandbox.task[task_id] = task_input_params
        logging.info('Created task https://sandbox.yandex-team.ru/task/{}'.format(task_id))

        resp = self._sandbox.batch.tasks.start.update(task_id)[0]
        if resp['status'] != 'SUCCESS':
            raise Exception('Failed to create task: {}'.format(resp['status']))

        return task_id

    def get_task_resource_skynet_id(self, task_id, resource_type):
        resources = self._sandbox.task[task_id].resources.read()
        for item in resources['items']:
            if item['type'] == resource_type:
                return item['skynet_id']

        raise ValueError(
            'Resource with type: {}, not found in {}'.format(resource_type, [r['type'] for r in resources['items']]))

    def get_task_resource_id(self, task_id, resource_type):
        resources = self._sandbox.task[task_id].resources.read()
        for item in resources['items']:
            if item['type'] == resource_type:
                return item['id']

        raise ValueError(
            'Resource with type: {}, not found in {}'.format(resource_type, [r['type'] for r in resources['items']]))

    def get_task_logs_url(self, task_id, log_path='debug.log'):
        resources = self._sandbox.task[task_id].resources.read()
        logs_http_proxy = None
        for item in resources['items']:
            if item['type'] == 'TASK_LOGS':
                logs_http_proxy = item['http']['proxy']

        if not logs_http_proxy:
            raise ValueError('Logs not found')

        return logs_http_proxy + "/{}".format(log_path)

    def get_task_data(self, task_id):
        return self._sandbox.task[task_id].read()

    def wait_task(self, task_id, idle_period=10):
        logging.info('Waiting for task to complete...')

        break_statuses = tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        while True:
            task_data = self._sandbox.task[task_id].read()
            status = task_data['status']
            yield status
            if status == 'SUCCESS' or status in break_statuses:
                break
            time.sleep(idle_period)

    def get_task_status(self, task_id):
        task_data = self._sandbox.task[task_id].read()
        return task_data['status']

    def release_task(self, task_id):
        task = self._sandbox.task[task_id].read()

        logging.info('About to release task {} ...'.format(task_id))
        self._sandbox.release({
            'task_id': task_id,
            'subject': task['description'],
            'type': 'stable',
        })
        logging.info('Released successfully')
