import json
import subprocess
import os
import sys
from collections import namedtuple
import logging
import urlparse
import requests

log = logging.getLogger('deploy-utils')
handler = logging.StreamHandler(sys.stdout)
handler.setLevel(logging.DEBUG)

log.addHandler(handler)


def get_arc_root():
    resp = subprocess.check_output(['arc', 'root'])
    return resp.strip()


def get_current_branch():
    resp = subprocess.check_output(['arc', 'branch', '--json'])
    branches = json.loads(resp)

    for b in branches:
        if b.get('current'):
            return b['name']

    raise RuntimeError('arc does not return branches')


BranchInfo = namedtuple('BranchInfo', ['name', 'commit', 'svn_revision'])


def get_branch_info(name):  # type: (str) -> BranchInfo
    resp = subprocess.check_output(['arc', 'branch', '--json', '-v'])
    branches = json.loads(resp)
    for b in branches:
        if b['name'] != name:
            continue

        commit = b['commit']['id']

        dump = subprocess.check_output(['arc', 'dump', 'object', commit])
        svn_revision = None
        for line in dump.split('\n'):
            if line.strip().startswith('revision:'):
                svn_revision = line.split('revision:')[1].strip()

        return BranchInfo(name=name, commit=commit, svn_revision=svn_revision)

    raise ValueError('branch with name {} not found'.format(name))


def get_svn_diff(left_commit, left_commit_svn_revision, right_commit, path):
    resp = subprocess.check_output(['arc', 'diff', left_commit, right_commit, path, '--svn'])

    return resp.replace('(rev 0)', '(rev {})'.format(left_commit_svn_revision))


def get_uncommitted_changes(path):  # type: (str) -> list
    resp = subprocess.check_output(['arc', 'status', path, '--json'])
    result = json.loads(resp)
    if result['status'].get('staged') or result['status'].get('changed'):
        return result['status'].get('staged', []) + result['status'].get('changed', [])
    return []


class PasteApiException(Exception):
    mute = True


def paste(text, file_type):
    PASTE_SERVER = 'https://paste.yandex-team.ru/'
    SESSION = requests.Session()
    payload = {
        'syntax': file_type,
        'text': text
    }
    r = SESSION.post(PASTE_SERVER, data=payload, allow_redirects=False)

    if r.status_code != 302:
        raise PasteApiException("Error uploading your paste")

    url = r.headers['Location']
    url_tuple = urlparse.urlparse(url)
    if url_tuple.netloc:
        return url
    server_tuple = urlparse.urlparse(PASTE_SERVER)
    url_tuple = (server_tuple[0], server_tuple[1],
                 url_tuple[2], url_tuple[3], url_tuple[4], url_tuple[5])
    return urlparse.urlunparse(url_tuple) + '/text'


PatchInfo = namedtuple('PatchInfo', ['trunk_svn_revision', 'paste_url', 'uncommitted_changes'])


def create_patch_for_sandbox_task(path):  # type: (str) -> tuple[bool, PatchInfo]
    arc_root = get_arc_root()

    dir_path = os.path.join(arc_root, path)
    if not os.path.exists(dir_path):
        raise ValueError('path should be relative arc root directory ({})'.format(arc_root))

    uncommitted_changes = get_uncommitted_changes(dir_path)

    if uncommitted_changes:
        return False, PatchInfo(None, None, uncommitted_changes)

    current_branch = get_current_branch()

    current_branch_info = get_branch_info(current_branch)

    trunk_branch_info = get_branch_info('trunk')

    diff = get_svn_diff(trunk_branch_info.commit, trunk_branch_info.svn_revision, current_branch_info.commit, dir_path)

    paste_url = paste(diff, 'diff')

    return True, PatchInfo(trunk_svn_revision=trunk_branch_info.svn_revision, paste_url=paste_url,
                           uncommitted_changes=None)


if __name__ == "__main__":
    create_patch_for_sandbox_task('infra')
