from __future__ import unicode_literals

import logging

import inject
from sepelib.core import config
from infra.qyp.novnc_server.contrib.websockify import websocketproxy

from infra.qyp.novnc_server.src import request_handler
from infra.qyp.novnc_server.src.plugins import auth_plugin
from infra.qyp.novnc_server.src.plugins import token_plugin
from infra.swatlib.auth import staff
from infra.swatlib.auth import passport
from infra.qyp.vmproxy.src import pod_controller, security_policy
from infra.qyp.vmproxy.src.lib import nanny_client
from infra.qyp.vmproxy.src.web import app as web_app

VERSION = 'development_version'


class Application(object):
    name = 'novnc_server'

    def init_web_application(self):
        default_yp_cluster = config.get_value('yp.default_cluster')
        sec_policy = security_policy.SecurityPolicy().from_sepelib_config()
        ctx = web_app.Ctx(
            nanny_client=nanny_client.NannyClient(**config.get_value('nanny')),
            vmagent_client=None,
            sec_policy=sec_policy,
            tvm_context=None,
            pod_ctl_factory=pod_controller.PodControllerFactory(default_yp_cluster, sec_policy),
            personal_quotas_dict=None,
            personal_accounts=set(),
            vmproxy_config=None,
        )
        novnc_cfg = config.get_value('novnc')

        return websocketproxy.WebSocketProxy(
            RequestHandlerClass=request_handler.CustomRequestHandler,
            source_is_ipv6=True,
            verbose=True,
            web=novnc_cfg.get('web'),
            auth_plugin=auth_plugin.VMAgentAuth(ctx=ctx),
            token_plugin=token_plugin.UnderscoredToken(),
            cert=novnc_cfg.get('cert'),
            key=novnc_cfg.get('key'),
            listen_port=novnc_cfg.get('port')
        )

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.log = logging.getLogger(self.name)
        self.web = self.init_web_application()
        inject.clear_and_configure(self.configure_injector)

    def configure_injector(self, binder):
        binder.bind(passport.IPassportClient, passport.PassportClient.from_config(config.get_value('passport')))
        binder.bind(staff.IStaffClient,
                    staff.StaffClient.from_config(config.get_value('staff')))

    def run(self):
        self.log.info('=' * 30)
        self.log.info('starting {}...'.format(self.name))
        # Block here
        self.web.start_server()

    def stop(self):
        self.log.info('stopping {}...'.format(self.name))
        self.log.info('=' * 30)


if __name__ == '__main__':
    import os
    path_to_config = os.path.abspath(__file__).split('/')[:-2]
    path_to_config.append('cfg_default.yml')
    config.load('/'.join(path_to_config))
    app = Application(123)
    app.run()
