import itertools
import json
import logging
import time

import requests
import ticket_parser2
import ticket_parser2.low_level


class QnotifierNotAuthorized(Exception):
    pass


class QNotifierClient(object):
    def __init__(self, secret):
        self.src = 2015617  # qdm
        self.dst = 2002376
        self.secret = secret

    def get_qnotifier_tvm_ticket(self):
        ts = int(time.time())

        tvm_keys = requests.get(
            'https://tvm-api.yandex.net/2/keys?lib_version=%s' % (ticket_parser2.__version__, )
        ).content.decode('utf-8')

        svc_ctx = ticket_parser2.low_level.ServiceContext(self.src, self.secret, tvm_keys)

        ticket_response = requests.post(
            'https://tvm-api.yandex.net/2/ticket/',
            data={
                'grant_type': 'client_credentials',
                'src': self.src,
                'dst': self.dst,
                'ts': ts,
                'sign': svc_ctx.sign(ts, self.dst)
            }
        ).json()

        ticket_for_dst = ticket_response[str(self.dst)]['ticket']
        return ticket_for_dst

    def notify_usergroups(self, tvm_ticket, users, groups, tags, subject, message):
        if not tvm_ticket:
            raise QnotifierNotAuthorized()

        next_is_group = object()
        is_group = False

        for item in itertools.chain(users, [next_is_group], groups):
            if item is next_is_group:
                is_group = True
                continue

            if is_group:
                logging.warning('Do not subscribing staff group %r (RTCSUPPORT-5708)', item)
                continue

            response = requests.post(
                'https://qnotifier.yandex-team.ru/subscriptions/%s' % (item if not is_group else 'group:%s' % (item, ), ),
                headers={
                    'X-Ya-Service-Ticket': tvm_ticket,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                data=json.dumps({
                    'tags': tags,
                    'options': {}
                })
            )

            if response.status_code in (401, 403):
                raise QnotifierNotAuthorized()

            if response.status_code == 500:  # seems to be ok, we need to log response headers + content for debug
                logging.warning('qnotifier subscribe returned HTTP500')
                logging.warning('headers: %r', response.headers)
                logging.warning('content: %r', response.content)
                raise Exception('Got HTTP500 during notification subscribtion')

            assert response.status_code in (204, 402), (
                'Invalid qnotifier response (code %d), was trying to subscribe %s (group: %s)' % (
                    response.status_code, item, is_group
                )
            )

        # message_markdown = []
        # for line in message.split('\n'):
        #     message_markdown.append('    %s' % (line, ))
        # message = '\n'.join(message_markdown)

        response = requests.post(
            'https://qnotifier.yandex-team.ru/events/',
            headers={
                'X-Ya-Service-Ticket': tvm_ticket,
                'Content-Type': 'application/json',
                'Accept': 'application/json',
            },
            data=json.dumps({
                'tags': tags,
                'message': message,
                'extra': {
                    'subject': subject,
                    'from_email': 'QYP <noreply@qyp.yandex-team.ru>',
                    # 'message-format': 'markdown'
                }
            })
        )

        if response.status_code == 500:  # seems to be ok, we need to log response headers + content for debug
            logging.warning('qnotifier subscribe returned HTTP500')
            logging.warning('headers: %r', response.headers)
            logging.warning('content: %r', response.content)
            raise Exception('Got HTTP500 during notification')

        assert response.status_code == 204, 'Unable to notify: status code %d' % (response.status_code, )
