import json
import logging

import requests


DC = {
    'vla': 'https://vla.yp.yandex.net:8443',
    'man': 'https://man.yp.yandex.net:8443',
    'sas': 'https://sas.yp.yandex.net:8443',
    'test_sas': 'https://sas-test.yp.yandex.net:8443',
    'myt': 'https://myt.yp.yandex.net:8443',
    'iva': 'https://iva.yp.yandex.net:8443',
    'man_pre': 'https://man-pre.yp.yandex.net:8443',
}


class YpClient(object):
    DEFAULT_TIMEOUT = 60

    def __init__(self, oauth_token, timeout=None):
        self.oauth_token = oauth_token
        self.timeout = timeout or self.DEFAULT_TIMEOUT
        self.headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': 'OAuth %s' % (self.oauth_token, )
        }

    def _api_call(self, url, data):
        result = requests.post(url, data=data, headers=self.headers, timeout=self.timeout)
        if result.status_code != 200:
            logging.warning('yp request failed with code: %d', result.status_code)
            logging.debug('response headers: %r', result.headers)

            if 'x-yt-response-message' in result.headers:
                raise Exception('YP Error: %s' % (result.headers['x-yt-response-message'],))
            else:
                raise Exception('YP request error (status code %d)' % (result.status_code,))
        return result

    def select_pods(self, dc, query):
        request = {
            'object_type': 'pod',
            'filter': {
                'query': query
            },
            'selector': {
                'paths': [
                    '/meta/id',                     # 0
                    '/meta/pod_set_id',             # 1
                    '/status/dns',                  # 2
                    '/status/scheduling/node_id',   # 3
                    '/status/eviction',             # 4
                    '/status/ip6_address_allocations',  # 5
                    '/labels',                      # 6
                    '/annotations/owners',          # 7
                    '/spec/ip6_address_requests',   # 8
                    '/spec/resource_requests',      # 9
                    '/spec/disk_volume_requests',   # 10
                    '/spec/iss/instances/0/properties',  # 11
                    '/status/scheduling/last_updated',  # 12
                    '/status/maintenance',  # 13
                    '/status/node_alerts',  # 14
                ]
            },
            'format': 1,  # PF_YSON
            'options': {
                'fetch_timestamps': True
            }
        }

        url = '%s/ObjectService/SelectObjects' % (DC[dc], )
        result = self._api_call(url, json.dumps(request))

        pods_for_eviction = []

        json_result = result.json()

        if 'results' not in json_result:
            # No pods for eviction as of now
            return pods_for_eviction

        for item in json_result['results']:
            item['values'] = [i['yson'] for i in item['value_payloads']]
            disk_volumes = item['values'][10]
            requested_volume = None
            for volume in disk_volumes:
                if volume.get('labels', {}).get('mount_path', None) == '/qemu-persistent':
                    requested_volume = volume
                    break

            network_ids = set()
            net_requests = item['values'][8]
            for net in net_requests:
                if net.get('labels', {}).get('owner', None) == 'vm':
                    network_ids.add(net['network_id'])

            assert len(network_ids) == 1
            network_id = network_ids.pop()

            # Detect nat64 if it is set in iss properties
            iss_properties = item['values'][11]
            use_nat64 = False

            for iss_prop in iss_properties:
                if iss_prop['key'] == 'USE_NAT64':
                    assert iss_prop['value'] == 'true'
                    use_nat64 = True
                    break

            # Detect tun v4/v6 if pod has internet address allocation
            address_allocations = item['values'][5]
            use_tun64 = False
            for addressinfo in address_allocations:
                if addressinfo['vlan_id'] == 'backbone':
                    if addressinfo.get('internet_address', None):
                        use_tun64 = True
                        break

            pods_for_eviction.append({
                'id': item['values'][0],
                'pod_set_id': item['values'][1],
                'fqdn': [
                    item['values'][2]['transient_fqdn'],
                    item['values'][2]['persistent_fqdn']
                ],
                'node_id': item['values'][3],
                'eviction': item['values'][4],
                'labels': item['values'][6],
                'owners': {
                    'users': item['values'][7]['logins'],
                    'groups': item['values'][7]['groups']
                },
                'requests': {
                    'net': {'network_id': network_id},
                    'resource': item['values'][9],
                    'disk_volume': requested_volume,
                },
                'use_nat64': use_nat64,
                'use_tun64': use_tun64,
                'scheduling_last_updated': item['values'][12],
                'maintenance': item['values'][13],
                'node_alerts': item['values'][14],
                'maintenance_ts': item['timestamps'][13],
            })

        return pods_for_eviction

    def acknowledge_maintenance(self, dc, pod_id, timestamp):
        request = {
            'object_type': 'pod',
            'object_id': pod_id,
            'set_updates': [{
                'path': '/control/acknowledge_maintenance',
                'value': {'message': 'Acknowledged by QDM'},
            }],
            'attribute_timestamp_prerequisites': [{
                'path': '/status/maintenance',
                'timestamp': timestamp,
            }]
        }

        url = '%s/ObjectService/UpdateObject' % (DC[dc],)
        self._api_call(url, json.dumps(request))
        return True
