import logging
import platform
import time
import traceback

import gevent
import requests

from .. import consts
from ..component import Component
from ..model import Evoq
from ..deblock import Deblock
from ..utils import LogDbObj

from .common import get_pods_for_eviction
from .process import EvoqProcess


class EvoqManager(Component):
    def __init__(self, db, dbt, qnotifier_client, yp_client, vmproxy_client, max_active_jobs,
                 disable_vm_leaving=None, parent=None):
        self.db = db
        self.dbt = dbt
        self.node = platform.node()
        self.qnotifier_client = qnotifier_client
        self.vmproxy_client = vmproxy_client
        self.yp_client = yp_client

        super(EvoqManager, self).__init__(parent=parent)

        self._logger = logging.getLogger('evoqmngr')
        self.log = LogDbObj(self.db, self._logger, 'evoqmngr')

        self._deblocker = Deblock(logger=self._logger.getChild('deblock'), keepalive=None)
        self.active = {}  # evoq_id => evoq

        self._max_active_jobs = max_active_jobs
        self.disable_vm_leaving = disable_vm_leaving or False

    def scheduler(self):
        # Schedule new evoq jobs
        self.log.debug('Scheduler run')

        deadline = time.time() + 600
        run_ticker = False

        active_evoqs = set()

        for dc in consts.DC_LIST:
            while True:
                try:
                    pods = self._deblocker.apply(
                        get_pods_for_eviction, dc, self.yp_client
                    )
                except requests.exceptions.ReadTimeout as ex:
                    time_left = max(0, int(deadline - time.time()))

                    self.log.warning(
                        'Timeout while trying to grab pods for eviction: %s, will retry (%s secs more)',
                        ex, time_left
                    )

                    if time_left < 1:
                        raise

                    gevent.sleep(min(time_left, 10))
                    continue
                else:
                    break

            ignore_pod_idxs = []

            for idx, pod_info in enumerate(pods):
                self.log.info('[%s][%s] Analyzing eviction request: %r %r', dc, pod_info['id'],
                              pod_info['maintenance'], pod_info['eviction'])
                if self._ignore_eviction(pod_info, dc):
                    ignore_pod_idxs.append(idx)
                else:
                    self.log.info('  pod eviction reason looks reasonable')

            for idx in reversed(ignore_pod_idxs):
                del pods[idx]

            try:
                for pod in pods:
                    vm_id = '%s.%s' % (pod['id'], dc)
                    node_id = pod['node_id']

                    self.log.info('Found pending evacuation request: %s from %s', vm_id, node_id)

                    active_evoqs.add(vm_id)

                    evoq_id = Evoq.find(self.db, vm_id, node_id)

                    if evoq_id:
                        # We already have this record in db, do not do anything for now
                        evoq = Evoq(self.db, vm_id, node_id)
                        assert evoq.load(evoq_id), 'Unable to find evoq record in db!'

                        self.log.info('  found existing evoq db record (id=%s, state=%s)', evoq.id, evoq.state)

                        if evoq.state == 'fail':
                            evoq.state = 'init'
                            evoq.session_key = None  # force reupload if it was already (partially?) done
                            evoq.save()

                            self.log.info('    job was failed, forced INIT state again')

                            run_ticker = True

                        elif evoq.state == 'stop':
                            # Do not retry stopped jobs, so no-op here
                            pass
                    else:
                        # Make new record
                        evoq = Evoq(self.db, vm_id, node_id)
                        evoq.active = True
                        evoq.init_ts = int(time.time())
                        evoq.run_node = self.node
                        evoq.run_cnt = 0
                        evoq.run_ts = int(time.time())
                        evoq.run_duration = 0
                        evoq.fail_cnt = 0
                        evoq.save()

                        self.log.info('  created new evoq record (id=%s, state=%s)', evoq.id, evoq.state)

                        run_ticker = True

            finally:
                pass

        self.log.info('Searching for active evoqs in db (currently really active %d)', len(active_evoqs))

        for active_evoq in Evoq.grab_active(self.db, self.node):
            if active_evoq.vm_id not in active_evoqs:
                self.log.info(
                    '  unable to find eviction request for active evacuation of %r, will set active=False',
                    active_evoq.vm_id
                )
                active_evoq.active = False
                active_evoq.save()
            else:
                self.log.info(
                    '  found active evoq record %r, keeping as is', active_evoq.vm_id
                )

        if run_ticker:
            self.db.execute(
                'UPDATE job SET '
                '   next_run_ts = %s, '
                '   last_node = %s '
                'WHERE name = %s', (int(time.time()), self.node, 'evoq_ticker')
            )

    def _ignore_eviction(self, pod_info, dc):
        if pod_info['maintenance'] and pod_info['maintenance']['state'] == 'requested':
            return False

        if not pod_info['eviction']:
            return False

        hfsm_eviction = pod_info['eviction']['reason'] == 'hfsm'
        if hfsm_eviction:
            return True

        reason = pod_info['eviction']['reason']
        hs_eviction = (reason == 'scheduler'
                       or reason == 'client' and 'cluster defragmentation' in pod_info['eviction']['message'])
        if hs_eviction:
            # QEMUKVM-1116
            scheduled_secs_ago = int(time.time() - (pod_info['scheduling_last_updated'] / 10 ** 6))
            scheduled_days_ago = int(scheduled_secs_ago / 86400)
            if dc in consts.CAN_IGNORE_HS_EVICTION_CLUSTERS or pod_info['id'] in consts.HS_EVICTION_VM_WHITELIST:
                max_days_limit = 180
            else:
                max_days_limit = 1
            if scheduled_days_ago < max_days_limit:
                self.log.info(
                    '  pod scheduled %d days (%d secs) ago, <30, ignoring eviction request',
                    scheduled_days_ago, scheduled_secs_ago
                )
                return True
            else:
                self.log.info('  pod scheduled %d days ago, eviction allowed', scheduled_days_ago)
                return False
        return False

    def ticker(self):
        self.log.debug('Ticker run')

        for evoq in Evoq.grab(self.db, self.node):
            self.log.info(
                'Grabbing evoq ticker #%s (vm %s, node %s, run_node %s)',
                evoq.id, evoq.vm_id, evoq.node_id, evoq.run_node
            )

            if evoq.id not in self.active:
                if len(self.active) < self._max_active_jobs:
                    if evoq.run_node == self.node:
                        self.log.info('  not running, readopting (same node)')
                    else:
                        self.log.info('  not running, stealing from %s', evoq.run_node)

                    self.active[evoq.id] = gevent.spawn(self._run_evoq, evoq.id, evoq)
                else:
                    if evoq.state == 'init':
                        self.log.info('  too much jobs (%d), moving state init => wait', len(self.active))
                        evoq.state = 'wait'
                        evoq.save()
                    else:
                        # It could be run or wait
                        #  run: we cant stop already running jobs
                        #  wait: already in waiting state
                        # Thus, no-op in this case
                        self.log.info('  too much jobs (%d), leaving job as is (%s)', len(self.active), evoq.state)

            else:
                self.log.info('  already running here')

    def _evoq_job_hold(self, key):
        started = time.time()

        while True:
            try:
                self.db.execute(
                    'UPDATE evoq '
                    'SET '
                    '   run_node = %s, '
                    '   run_ts = %s, '
                    '   run_duration = run_duration + %s '
                    'WHERE id = %s', (
                        self.node, int(time.time()), int(time.time() - started), key
                    )
                )
                started = time.time()
                gevent.sleep(60)

            except gevent.GreenletExit:
                self.db.execute(
                    'UPDATE evoq SET run_duration = run_duration + %s WHERE id = %s',
                    (int(time.time() - started), key)
                )

                self.log.debug('job holdr got GreenletExit')
                return

            except:
                self.log.warning('Unable to update ticker')

    def _run_evoq(self, key, evoq):
        try:
            current_grn = gevent.getcurrent()

            holder = gevent.spawn(self._evoq_job_hold, key)

            # Link each other's exit
            holder.link(lambda grn: current_grn.kill())
            current_grn.link(lambda grn: holder.kill())

            self.log.info('Running EvoqProcess')
            process = EvoqProcess(
                log=logging.getLogger('evoqproc'),
                db=self.db,
                key=key,
                evoq=evoq,
                qnotifier_client=self.qnotifier_client,
                vmproxy_client=self.vmproxy_client,
                yp_client=self.yp_client,
                disable_vm_leaving=self.disable_vm_leaving
            )
            process.run()

        except gevent.GreenletExit:
            self.log.warning('Got GreenletExit on run_evoq, probably holdr died')
            return

        except Exception as ex:
            self.log.warning('EvoqProcess died with: %s: %s', type(ex).__name__, ex)
            self.log.warning(traceback.format_exc())
            evoq.state = 'fail'
            evoq.save()
            # There is no point to raise exception deeper, so just exit
            return

        else:
            evoq.state = 'done'
            evoq.save()
            self.log.info('EvoqProcess finished without errors')

        finally:
            self.active.pop(key)
