import datetime
import gevent
import json
import logging
import os
import platform
import random
import time
import urllib

import requests

from . import consts
from .component import Component
from .deblock import Deblock
from .evoq.common import DC
from .utils import LogDbObj, human_secs
from .model import Session

from .evoq.common import select_pods


class HotBackup(Component):
    def __init__(self, db, dbt, tvm_secret, yp_oauth_token, vmproxy_oauth_token, parent=None):
        self.db = db
        self.dbt = dbt
        self.node = platform.node()
        self.tvm_secret = tvm_secret
        self.yp_oauth_token = yp_oauth_token
        self.vmproxy_oauth_token = vmproxy_oauth_token

        super(HotBackup, self).__init__(parent=parent)

        self._logger = logging.getLogger('hotbackup')
        self.log = LogDbObj(self.db, self._logger, 'hotbackup')

        self._deblocker = Deblock(logger=self._logger.getChild('deblock'), keepalive=None)

        self._scheduler_grn = None

        # session = Session(self.db)
        # session.generate('upload', 'hot')
        # session.vm_id = 'mcsl4'
        # session.node_id = 'vla3-4278.search.yandex.net'
        # session.save()
        # complete, status_code, result = self._vmproxy_api_call(
        #     'vla', '/api/MakeAction/', {
        #         'vm_id': {'pod_id': 'mcsl4'},
        #         'action': 'QDMUPLOAD_BG',
        #         'qdmreq': {'key': session.key}
        #     }
        # )

    def scheduler(self):
        LIMIT_HOT_DAY = 40
        LIMIT_HOT_NIGHT = 80

        self.log.debug('Scheduler run')

        if self._scheduler_grn and self._scheduler_grn:
            self.log.critical('Old scheduler thread still alive -- exit immidiately!')
            os._exit(1)

        now = int(time.time())

        vms_ordered = self.db.query(
            'SELECT vm_id, node_id, age FROM ('
            '    SELECT'
            '        vm.vm_id as vm_id,'
            '        %s - MAX(sr.create_ts) as age,'  # will be NULL if vm has no backups, on top of the list
            '        MAX(vm.hot_period) as hot_period,'
            '        MAX(vm.node_id) as node_id'
            '    FROM vm'
            '    LEFT JOIN storage_revision sr ON (sr.vm_id = vm.vm_id)'
            '    WHERE ('
            '        ('
            '            vm.last_touch_ts < %s - 86400'
            '            OR ('
            '                vm.last_touch_ts < %s - 3600'
            '                AND vm.vm_id LIKE %s'
            '            )'
            '        )'
            '        AND vm.exists'
            '        AND vm.hot_allow'
            '    )'
            '    GROUP BY vm.vm_id'
            '    ORDER BY hot_period ASC, age DESC'
            ') as subq '
            'WHERE subq.age IS NULL OR subq.age >= subq.hot_period',
            (
                now,  # age
                now,  # last_touch_ts for regular vms
                now,  # last_touch_ts for test_sas
                '%.test_sas'
            )
        )
        # vms_ordered = [v for v in vms_ordered if v[0].endswith('.test_sas')]

        vms_without_backup = len([x for x in vms_ordered if x[2] is None])
        vms_with_backup = len([x for x in vms_ordered if x[2] is not None])

        active_sessions = self.db.query_one_col('SELECT COUNT(*) FROM session WHERE state = %s', ('active', ))
        is_night = datetime.datetime.now().hour >= 21 or datetime.datetime.now().hour < 9
        allow_to_run = (LIMIT_HOT_NIGHT if is_night else LIMIT_HOT_DAY) - active_sessions

        self.log.debug('Found %d total vms in cache', len(vms_ordered))
        self.log.debug('  %d with backups', vms_with_backup)
        self.log.debug('  %d without backups', vms_without_backup)
        self.log.debug('  %d active sessions', active_sessions)
        self.log.debug('  %d allowed to run', allow_to_run)

        # reorder vm wo backups first
        vms_reordered = [], []
        for vm_info in vms_ordered:
            if not vm_info[2]:
                vms_reordered[0].append(vm_info)
            else:
                vms_reordered[1].append(vm_info)

        random.shuffle(vms_reordered[0])
        vms_ordered = vms_reordered[0] + vms_reordered[1]

        if allow_to_run > 0:
            self._scheduler_grn = gevent.spawn(self.scheduler_bg, vms_ordered[:allow_to_run])

    def scheduler_bg(self, vms):
        # vm_list: vm_id, node_id, age of last backup

        for vm_id, node_id, last_backup_age_ts in vms:
            ts = time.time()
            try:
                self.hotbackup_vm(vm_id, node_id, last_backup_age_ts)
            except Exception as ex:
                import traceback
                self.log.warning('Unable to schedule hotbackup for vm %s: %s: %s', vm_id, type(ex).__name__, ex)
                self.log.warning(traceback.format_exc())
                gevent.sleep(1)  # avoid busy loops
            else:
                sleep = 0.1 - (time.time() - ts)
                if sleep > 0:
                    gevent.sleep(sleep)
                else:
                    gevent.sleep()  # yield

    def hotbackup_vm(self, vm_id, node_id, last_backup_age_ts):
        self.log.debug(
            'Attempt to hotbackup vm: %s, node %s, last_backup_age: %s',
            vm_id, node_id, human_secs(last_backup_age_ts) if last_backup_age_ts is not None else 'never'
        )

        now = int(time.time())

        # Update touch time before any other actions
        self.db.execute('UPDATE vm SET last_touch_ts = %s WHERE vm_id = %s', (now, vm_id))

        # Convert our vm_id to pod_id and dc
        pod_id, dc = vm_id.rsplit('.', 1)

        pods = self._deblocker.apply(
            select_pods, dc, self.yp_oauth_token, '[/meta/id] = "%s"' % (pod_id, )
        )

        if len(pods) == 0:
            self.log.info('YP reported 0 pods matching search criteria (dc: %s, pod_id: %s)', dc, pod_id)
            return False

        assert len(pods) == 1
        podinfo = pods[0]

        vmagent_version = podinfo.get('labels', {}).get('vmagent_version', '0.0')
        try:
            vmagent_version_major, vmagent_version_minor = vmagent_version.split('.', 1)
            vmagent_version_major = int(vmagent_version_major)
            vmagent_version_minor = int(vmagent_version_minor)
            vmagent_version = (vmagent_version_major, vmagent_version_minor)
        except:
            vmagent_version = (0, 0)

        if vmagent_version < (0, 34) and pod_id != 'mcsl2':
            self.log.info('  ignoring: vmagent is too old (%s)', vmagent_version)
            return False

        # ensure we have no active sessions and draft revisions for this vm

        active_sessions = self.db.query_one_col(
            'SELECT COUNT(*) FROM session WHERE vm_id = %s AND state = %s',
            (vm_id, 'active')
        )
        if active_sessions:
            self.log.info('  ignoring: already %d active sessions', active_sessions)
            return False

        draft_revisions = self.db.query_one_col(
            'SELECT COUNT(*) FROM storage_revision WHERE vm_id = %s AND state = %s',
            (vm_id, 'draft')
        )
        if draft_revisions:
            self.log.info('  ignoring: already %d draft (in progress) revisions', draft_revisions)
            return False

        session = Session(self.db)
        session.generate('upload', 'hot')
        session.vm_id = vm_id
        session.node_id = node_id
        session.save()

        complete, status_code, result = self._vmproxy_api_call(
            dc, '/api/MakeAction/', {
                'vm_id': {'pod_id': pod_id},
                'action': 'QDMUPLOAD_BG',
                'qdmreq': {'key': session.key}
            }
        )

        if not complete:
            self.log.error(
                '  unable to execute QDMUPLOAD_BG (code %r, response %r)',
                status_code, result
            )
        else:
            self.log.info('  QDMUPLOAD_BG scheduled successfully')

    def _vmproxy_api_call(self, dc, url, data):
        url = urllib.parse.urljoin(DC[dc]['vmproxy'], url)

        response = self._deblocker.apply(
            requests.post, url, data=json.dumps(data),
            headers={
                'Content-Type': 'application/json',
                'Authorization': 'OAuth %s' % (self.vmproxy_oauth_token, )
            },
            timeout=60
        )

        if response.headers.get('Content-Type', '') == 'application/json':
            result = json.loads(response.text)
        else:
            result = response.text

        if response.status_code != 200:
            self.log.error('vmproxy request failed with code %d', response.status_code)
            self.log.error('vmproxy request output')
            self.log.error(response.text)

            return False, response.status_code, result
        return True, response.status_code, result

    def vmsync(self):
        self.log.debug('VM sync run')

        now = int(time.time())

        for dc in consts.DC_LIST:
            # for dc in ('man', ):
            self.log.info('Synchinizing VMs in DC %s', dc)
            pod_list = {}

            request = {
                'object_type': 'pod',
                'filter': {
                    'query': '[/labels/deploy_engine] = "QYP"',
                },
                'selector': {
                    'paths': [
                        '/meta/id',
                        '/meta/pod_set_id',
                        '/meta/creation_time',
                        '/spec/node_id',
                        '/labels/vmagent_version',
                        '/status/scheduling',
                    ]
                }
            }

            url = '%s/ObjectService/SelectObjects' % (DC[dc]['yp'], )
            result = self._deblocker.apply(
                requests.post, url, data=json.dumps(request),
                headers={
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                    'Authorization': 'OAuth %s' % (self.yp_oauth_token, )
                },
                timeout=60
            )

            if result.status_code != 200:
                self.log.warning('  yp request failed with code: %d', result.status_code)
                self.log.debug('  response headers: %r', result.headers)

                if 'x-yt-response-message' in result.headers:
                    raise Exception('YP Error: %s' % (result.headers['x-yt-response-message'], ))
                else:
                    raise Exception('YP request error (status code %d)' % (result.status_code, ))

            json_result_pods = result.json()

            request = {
                'object_type': 'pod_set',
                'filter': {
                    'query': '[/labels/deploy_engine] = "QYP"',
                },
                'selector': {
                    'paths': [
                        '/meta/id',
                        '/spec/node_segment_id'
                    ]
                }
            }

            result = self._deblocker.apply(
                requests.post, url, data=json.dumps(request),
                headers={
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                    'Authorization': 'OAuth %s' % (self.yp_oauth_token, )
                },
                timeout=60
            )

            if result.status_code != 200:
                self.log.warning('  yp request failed with code: %d', result.status_code)
                self.log.debug('  response headers: %r', result.headers)

                if 'x-yt-response-message' in result.headers:
                    raise Exception('YP Error: %s' % (result.headers['x-yt-response-message'], ))
                else:
                    raise Exception('YP request error (status code %d)' % (result.status_code, ))

            json_result_pod_sets = result.json()

            for item in json_result_pods['results']:
                pod_id, pod_set_id, creation_time, node_id, vmagent_version, scheduling_status = item['values']
                if pod_id != pod_set_id:
                    self.log.warning('  pod set ID != Pod ID: %s != %s', pod_id, pod_set_id)

                pod_list['%s.%s' % (pod_id, dc)] = {
                    'segment': None,
                    'node_id': node_id,
                    'vmagent_version': vmagent_version,
                    'creation_time': creation_time / (10**6),
                    'scheduling_state': scheduling_status['state'],
                }

            for item in json_result_pod_sets['results']:
                pod_set_id, segment_id = item['values']
                pod_set_id_with_dc = '%s.%s' % (pod_set_id, dc)

                if pod_set_id_with_dc in pod_list:
                    pod_list[pod_set_id_with_dc]['segment'] = segment_id

            for pod_id, pod_info in pod_list.items():
                if not pod_info['segment']:
                    self.log.warning('  unable to find pod segment for pod %r', pod_id)
                    pod_list.pop(pod_id)
                    continue

            with self.dbt:
                stats = {
                    'added': 0,
                    'removed': 0,
                    'updated': 0,
                    'ok': 0
                }

                pod_list_in_db = {}
                for vm_id, segment, node_id, exists, hot_allow, hot_period in self.dbt.query(
                    'SELECT vm_id, segment, node_id, exists, hot_allow, hot_period FROM vm WHERE vm_id LIKE %s',
                    ('%%.%s' % (dc, ), )
                ):
                    pod_list_in_db[vm_id] = {
                        'segment': segment,
                        'node_id': node_id,
                        'exists': exists,
                        'hot_allow': hot_allow,
                        'hot_period': hot_period
                    }

                for vm_id, vm_info in pod_list.items():
                    vmagent_version = [int(x) for x in vm_info['vmagent_version'].split('.', )]

                    vmagent_version_valid = vmagent_version >= [0, 34]
                    segment_valid = (vm_info['segment'] in ('dev', 'gpu-dev')
                                     or dc in ('test_sas', )
                                     or dc == 'man' and vm_id.startswith('jupyter-cloud-'))
                    if vmagent_version_valid and segment_valid:
                        hot_allow = True
                    else:
                        hot_allow = False

                    if vm_info['scheduling_state'] != 'assigned':
                        hot_allow = False

                    if vm_info['creation_time'] > (now - 7 * 86400):
                        hot_allow = False

                    if vm_id not in pod_list_in_db:
                        hot_period = 86400 * 7
                        self.dbt.execute(
                            'INSERT INTO vm ('
                            '   vm_id, segment, node_id, '
                            '   exists, last_touch_ts, update_ts, hot_allow, hot_period'
                            ') VALUES (%S)',
                            (
                                (
                                    vm_id, vm_info['segment'], vm_info['node_id'],
                                    True, 0, now, hot_allow, hot_period
                                ),
                            )
                        )
                        if stats['added'] < 100:
                            self.log.debug('  add %s', vm_id)
                        elif stats['added'] == 100:
                            self.log.debug('  too much new machines, we will not log them anymore')

                        stats['added'] += 1
                    else:
                        vm_info_in_db = pod_list_in_db[vm_id]
                        update_reasons = []

                        if vm_info['segment'] != vm_info_in_db['segment']:
                            update_reasons.append(
                                'segment changed %s => %s' % (vm_info_in_db['segment'], vm_info['segment'])
                            )
                        if vm_info['node_id'] != vm_info_in_db['node_id']:
                            update_reasons.append(
                                'node changed %s => %s' % (vm_info_in_db['node_id'], vm_info['node_id'])
                            )
                        if hot_allow != vm_info_in_db['hot_allow']:
                            update_reasons.append(
                                'hot_allow changed %s => %s' % (vm_info_in_db['hot_allow'], hot_allow)
                            )
                        if not vm_info_in_db['exists']:
                            update_reasons.append(
                                'mark as existent'
                            )

                        if update_reasons:
                            self.log.debug('  update %s (reason: %s)', vm_id, ', '.join(update_reasons))
                            self.dbt.execute(
                                'UPDATE vm SET '
                                '   segment = %s, '
                                '   node_id = %s, '
                                '   exists = %s, '
                                '   update_ts = %s, '
                                '   hot_allow = %s '
                                'WHERE vm_id = %s',
                                (
                                    vm_info['segment'], vm_info['node_id'], True, now,
                                    hot_allow,
                                    vm_id
                                )
                            )
                            stats['updated'] += 1
                        else:
                            stats['ok'] += 1

                for vm_id, vm_info in pod_list_in_db.items():
                    if vm_info['exists'] and vm_id not in pod_list:
                        self.log.debug('  mark %s as non existent anymore', vm_id)
                        self.dbt.execute(
                            'UPDATE vm SET exists = %s, update_ts = %s WHERE vm_id = %s',
                            (False, now, vm_id)
                        )
                        stats['removed'] += 1

                self.log.info(
                    '  final steps -- complete (added %d, updated %d, removed %d, up to date %d)',
                    stats['added'], stats['updated'], stats['removed'], stats['ok']
                )
